#ifndef mobio_parser_h_
#define mobio_parser_h_

//:
// \file
// \brief Classes for parsing MoBio project data structures.
// \author Tim Cootes

#include <cstdlib>
#include <cstring>
#include <vector>
#include <iostream>
#include <string>

//: Read a block of text from a stream.
// This function will read through a stream, and store the text found to a string.
// The function terminates when it finds the closing brace.
//
// The stream's fail bit will be set on error. 
// Comment lines beginning with "//" will be stripped.
// \param open_already should be true if the client has already
// read the opening brace. 
// If set to false and the first non-whitespace character is
// not an opening brace then that character will be left in the stream and "{}" will be returned.
// \return the text including the opening and closing braces.
// \param comment Lines beginning with white space followed by this string will be ignored.
std::string mobio_parse_block(std::istream &is, 
                           bool open_already = false,
                           const char * comment = "//");

//: Extracts list of strings from stream
//  Expects format "{ str1 str2 str3 ... }".
//  Aborts if fails to find opening or closing brace.
void mobio_parse_string_list(std::istream& is,
                             std::vector<std::string>& str_list);


//: Class to indicate indentation when writing text
class mobio_indent {};

//: Writes a number of spaces depending on current indent for stream
std::ostream& operator<<(std::ostream& os, const mobio_indent&);

//: Increase indentation associated with given stream
void mobio_inc_indent(std::ostream& os);

//: Decrease indentation associated with given stream
void mobio_dec_indent(std::ostream& os);


// ==================================================================
// INPUTS
// ==================================================================

/*
//: Structure to hold parameters of specific video file
struct mobio_videodata
{
  std::string filename; // pretty redundant
  double frame_length;
  unsigned start_frame;
  unsigned end_frame;

  //: Default constructor
  mobio_videodata()
    : frame_length(0.0),start_frame(0),end_frame(0) {}
};

//: Parses stream for video parameters
// Expects format "filename: ..."
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        mobio_videodata& f);

//: Writes video parameters to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_videodata& f);

*/

// ==================================================================

//: Structure specifying which videos are associated with each person
struct mobio_id
{
  std::string name;
  std::vector<std::string> videos;

  //: Default/utility constructor
  mobio_id(std::string namestr = "<blank>") : name(namestr) {}
};

//: Parses stream for ID
// Expects format "{ name: ... videos: ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_id& id);

//: Parses stream for set of IDs
// Expects format "{ ID: { name: ... } ID: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_id>& vec_id);

//: Writes ID to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_id& id);

//: Writes set of IDs to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_id>& vec_id);


// ==================================================================

//: Structure specifying which videos are tested against each ID
struct mobio_test
{
  mobio_id actual_id;
  std::vector<std::string> claimed_ids;

  //: Default constructor
  mobio_test() {}
};

//: Parses stream for test
// Expects format "{ ID: { ... } challenge: { ... } }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_test& test);

//: Parses stream for set of test
// Expects format "{ test: { ... } test: { ... } }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_test>& vec_test);

//: Writes test to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_test& test);

//: Writes set of tests to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_test>& vec_test);


// ==================================================================

//: Structure to hold details of which files to process and output
struct mobio_inputfile
{
  std::string sparams_file;
  std::string input_dir;
  std::string model_dir;
  std::string output_dir;
  std::string file_ext;
  std::vector<mobio_id> ids;
  std::vector<mobio_test> tests;

  // Default constructor
  mobio_inputfile()
    : sparams_file("<blank>"), input_dir("."), model_dir("."), output_dir("."), file_ext("avi") {}

  // Return list of all videos
  std::vector<std::string> all_videos();

  // Return list of all enrolled ID
  std::vector<std::string> all_ids();

  // check that all identities to be claimed in tests are valid
  bool valid_claimed_ids();
};

//: Parses stream for input parameters (directories, IDs and datasets)
// Expects format "video_dir: ... "
// Reads to the end of the file, or to next unread closing brace "}"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        mobio_inputfile& f);

//: Writes parameters to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_inputfile& f);



// ==================================================================
// OUTPUTS
// ==================================================================

//: Summarises output of face detector
// Centre of box is at (cx,cy)
// Width wx, height wy.
// In plane rotation az
// Rotation about x axis (nodding) ax
// Rotation about y axis (left/right), ay
struct mobio_facebox
{
  double cx;
  double cy;
  double wx;
  double wy;

  //: Rotation about x axis (nodding)
  double ax;

  //: Rotation about y axis (left/right)
  double ay;

  //: In plane rotation
  double az;

  //: Quality score (high is good)
  double score;

  //: Default constructor
  mobio_facebox()
    : cx(0),cy(0),wx(0),wy(0),ax(0),ay(0),az(0),score(0) {}

  //: Utility constructor
  mobio_facebox(double cx1, double cy1, double wx1, double wy1,
                double score1);
};

//: Parses stream for facebox
// Expects format "{ cx: ... cy: ... wx: ... wy: ... score: ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_facebox& fb);

//: Parses stream for sets of faceboxes (one frame)
// Expects format "{ facebox: { cx: ... } facebox: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_facebox>& vec_fb);

//: Parses stream for sets of sets of faceboxes (all frames)
// Expects format "{ frame: { facebox: ... } frame: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector< std::vector<mobio_facebox> >& frames);

//: Writes facebox to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_facebox& fb);

//: Writes set of faceboxes to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_facebox>& vec_fb);

//: Writes set of set of faceboxes to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector< std::vector <mobio_facebox> >& frames);


// ==================================================================
struct mobio_video_faceboxes
{
  //: Video filename
  std::string filename;

  //: Length of each frame in ms
  double frame_length;

  //: Index of first frame of interest
  unsigned start_frame;

  //: frame_data[f][i] gives the i-th face on frame (f+start_frame)
  std::vector< std::vector <mobio_facebox> > frame_data;

  //: Default constructor
  mobio_video_faceboxes()
    : filename("<blank>"),frame_length(0.0),start_frame(0) {}
};

//: Writes mobio_video_faceboxes to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_video_faceboxes& vfb);

//: Parses stream for mobio_video_faceboxes
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        mobio_video_faceboxes& vfb);

// ==================================================================


struct mobio_facept
{
  double x,y,score;

  //: Default constructor
  mobio_facept()
    : x(0),y(0),score(0) {}

  //: Utility constructor
  mobio_facept(double x1,double y1,double s1=0.0)
    : x(x1),y(y1),score(s1) {}
};

//: Parses stream for facept
// Expects format "{ x: ... y: ... score: ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        mobio_facept& pt);

//: Parses stream for set of facepts (one face)
// Expects format "{ p: { x: ... } p: { ... } p: { ... } }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_facept>& vec_pt);

//: Writes facept to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_facept& pt);

//: Writes set of facepts to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_facept>& vec_pt);


// ==================================================================

//: Stores a set of points and a score associated with a single face
struct mobio_face_points
{
  std::vector<mobio_facept> points;
  double score;

  //: Default constructor
  mobio_face_points()
    : score(0) {}

  // return eye centres
  std::vector<double> eye_centres();
};

//: Parses stream for face
// Expects format "{ score: 0.87 pts: { x: ... } }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        mobio_face_points& face);

//: Parses data for set of faces (one frame)
// Expects format "{ face: { score: ... } face: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_face_points>& vec_face);

//: Parses stream for set of set of faces (all frames)
// Expects format "{ frame: { face: ... } frame: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector< std::vector<mobio_face_points> >& frames);

//: Writes face to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_face_points& face);

//: Writes set of faces to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_face_points>& vec_face);

//: Writes set of set of faces to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector< std::vector<mobio_face_points> >& frames);

// ==================================================================
struct mobio_video_face_points
{
  //: Video filename
  std::string filename;

  //: Length of each frame in ms
  double frame_length;

  //: Index of first frame of interest
  unsigned start_frame;

  //: frame_data[f][i] gives the i-th face on frame (f+start_frame)
  std::vector< std::vector <mobio_face_points> > frame_data;

  //: Default constructor
  mobio_video_face_points()
    : filename("<blank>"),frame_length(0.0),start_frame(0) {}
};

//: Writes mobio_video_face_points to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_video_face_points& vfp);

//: Parses stream for mobio_video_face_points
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        mobio_video_face_points& vfp);

// ==================================================================

struct mobio_segment
{
  double score;

  //: Default constructor
  mobio_segment(double scr = 0.0) : score(scr) {}
};

//: Parses stream for score
// Expects format "{ score: ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_segment& segment);

//: Parses stream for set of segments
// Expects format "{ segment: { score: ... } segment: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_segment>& vec_segment);

//: Writes segment to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_segment& segment);

//: Writes set of segments to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_segment>& vec_segment);

// ==================================================================

//: Structure specifying the result of an attempted authentication
struct mobio_result
{
  std::string claimed_id;
  double total_score;
  double segment_length;
  double start_time;
  std::vector<mobio_segment> scores;

  //: Default constructor
  mobio_result()
    : claimed_id("<blank>"),total_score(0.0),segment_length(0.0),start_time(0.0) {}
};

//: Parses stream for result
// Expects format "{ claimed_id: ... total_score: ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_result& result);

//: Parses stream for set of results
// Expects format "{ result: { claimed_id: ... } result: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_result>& vec_result);

//: Writes result to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_result& result);

//: Writes set of results to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_result>& vec_result);

// ==================================================================

//: Output of a set of attempted authentications
struct mobio_outputfile
{
  std::string filename;
  std::string actual_id;
  std::vector<mobio_result> results;

  //: Default constructor
  mobio_outputfile()
    : filename("<blank>"),actual_id("<blank>") {}
};

//: Parses stream for output file
// Expects format "{ filename: ... actual_id: ... results: { ... } }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_outputfile& outputfile);

//: Writes output file to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_outputfile& outputfile);

// ==================================================================

#endif //  mobio_parser_h_


