// Example of feature locator tool, demonstrating use of interfaces

// N.B. This example is written using the VXL package

#include <fstream>           // Standard file stream classes
#include "mobio_parser.h"    // mobio data classes + parsing code

// Strip the final ".xxx" from s
std::string strip_extension(const std::string& s)
{
  size_t p = s.find_last_of(".");
  if (p==std::string::npos) return s;  // No "." found
  return s.substr(0,p);
}

// Reads in video from video_name in video_dir
// Processes it to find faces
// Writes output to file in output_dir
void conduct_test(const mobio_test& test,
                  const std::string& input_dir,
                  const std::string& output_dir,
                  const std::string& file_ext)
{
  // process each candidate video separately
  for (unsigned i=0; i<test.actual_id.videos.size(); i++)
  {
    std::string video_name = test.actual_id.videos[i];

    // create output structure for each input video
    mobio_outputfile outputfile;
    outputfile.actual_id = test.actual_id.name;
    outputfile.filename = video_name;

    std::string video_path = input_dir+video_name+'.'+file_ext;

    std::cout<<"Processing video "<<video_path<<std::endl;

    std::string basepath = output_dir+video_name;


    // Load video from video_path here
    // ...


    // Read in list of face boxes recorded for this video
    mobio_video_face_points facepoints;
    std::string facepoint_path = basepath +"_facepts.txt";
    std::ifstream ifs(facepoint_path.c_str());
    if (!ifs)
    {
      std::cerr<<"Unable to open facepoint data from "<<facepoint_path<<std::endl;
      std::cerr<<"Have you run the feature localizer tool?"<<std::endl;
      abort();
    }
    ifs>>facepoints;
    ifs.close();


    // for each claimed_id (c)...
    for (unsigned c=0; c<test.claimed_ids.size(); c++)
    {
      // ...authenticate against video (i)
      mobio_result result;
      result.claimed_id = test.claimed_ids[c];
      result.segment_length = 40;
      result.start_time = 0.0;
      
      // pretend there are 5 segments for now
      int n_segments = 5;
      for (int s=0; s<n_segments; s++)
      {
        // evaluate each segment in turn and assign a score (e.g. 0.42)
        mobio_segment segment(0.42); // e.g. score=0.42
        result.scores.push_back(segment);
      }

      //update total score based on individual scores
      result.total_score = 0.87; // e.g. 0.87

      // add to outputfile
      outputfile.results.push_back(result);
    }

    // Save the results to output directory
    std::string out_path = output_dir+video_name+"_results.txt";
    std::ofstream ofs(out_path.c_str());
    if (!ofs)
    {
      std::cerr<<"Unable to open file "<<out_path<<" for output."<<std::endl;
      abort();
    }
    ofs<<outputfile;
    ofs.close();
  }
}

void print_usage()
{
  std::cout<<"Example of face authentication tool, demonstrating use of interfaces"<<std::endl;
  std::cout<<"Usage: mobio_fa_uman_stl_v1 -i <inputfile>"<<std::endl;
}

int main(int argc, char** argv)
{
  if (argc<3)
  {
    print_usage();
    return 0;
  }

  // Process the command line
  std::string input_path(argv[2]);

  // Read in parameters by parsing input text file
  mobio_inputfile inputfile;
  std::ifstream ifs(input_path.c_str());
  ifs>>inputfile;

  std::cout<<"Input data: "<<std::endl<<inputfile<<std::endl;

  // Now do each authentication test
  // perform each test in inputfile
  for (unsigned t=0; t<inputfile.tests.size(); t++)
    conduct_test(inputfile.tests[t],
                 inputfile.input_dir,
                 inputfile.output_dir,
                 inputfile.file_ext);
}
