// Example of feature locator tool, demonstrating use of interfaces

// N.B. This example is written using the VXL package

#include <fstream>           // Standard file stream classes
#include "mobio_parser.h"    // mobio data classes + parsing code

// Strip the final ".xxx" from s
std::string strip_extension(const std::string& s)
{
  size_t p = s.find_last_of(".");
  if (p==std::string::npos) return s;  // No "." found
  return s.substr(0,p);
}

// Reads in video from video_name in video_dir
// Processes it to find faces
// Writes output to file in output_dir
void process_video(const std::string& video_dir,
                   const std::string& output_dir,
                   const std::string& video_name)
{
  std::string video_path = video_dir+video_name;
  std::cout<<"Processing video "<<video_path<<std::endl;

  std::string basepath = output_dir+strip_extension(video_name);

  // Read in list of face boxes recorded for this video
  mobio_video_faceboxes faceboxes;
  std::string facebox_path = basepath +"_faceboxes.txt";
  std::ifstream ifs(facebox_path.c_str());
  if (!ifs)
  {
    std::cerr<<"Unable to open facebox data from "<<facebox_path<<std::endl;
    std::cerr<<"Have you run the face detector tool?"<<std::endl;
    abort();
  }
  ifs>>faceboxes;
  ifs.close();

  // Load video from video_path
  // ...

  unsigned n_frames=5;  // Set to number of frames in video
  
  // Set up structure to hold results
  mobio_video_face_points results;
  results.filename = video_name;
  results.frame_length = faceboxes.frame_length; // Or whatever
  results.start_frame = faceboxes.start_frame;

  unsigned i0=results.start_frame;
  for (unsigned i=i0;i<n_frames;++i)
  {
    std::vector<mobio_face_points> frame_data;

    // Process frame i, searching each face box in turn for 
    // a set of facial feature points
    unsigned n_boxes = faceboxes.frame_data[i-i0].size();
    for (unsigned j=0;j<n_boxes;++j)
    {
      const mobio_facebox& box = faceboxes.frame_data[i-i0][j];

      // Find feature points in face given box
      // ...
      // For example:
      mobio_face_points face_points;
      face_points.score = 0.79;  // Quality of whole face fit
      // Create two points as a dummy
      mobio_facept p1(box.cx-0.5*box.wx,box.cy-0.5*box.wy,0.5);
      mobio_facept p2(box.cx+0.5*box.wx,box.cy+0.5*box.wy,0.5);
      face_points.points.push_back(p1);
      face_points.points.push_back(p2);

      frame_data.push_back(face_points);
    }

    // Store result for this frame
    results.frame_data.push_back(frame_data);

  }

  // Save the results to output directory
  std::string out_path = output_dir+strip_extension(video_name)
                          +"_facepts.txt";

  std::ofstream ofs(out_path.c_str());
  if (!ofs)
  {
    std::cerr<<"Unable to open file "<<out_path<<" for output."<<std::endl;
    abort();
  }
  ofs<<results;
  ofs.close();
}

void print_usage()
{
  std::cout<<"Example of feature locator tool, demonstrating use of interfaces"<<std::endl;
}

int main(int argc, char** argv)
{
  if (argc<3)
  {
    print_usage();
    return 0;
  }

  // Process the command line
  std::string input_path(argv[2]);

  // Read in parameters by parsing input text file
  mobio_inputfile data;
  std::ifstream ifs(input_path.c_str());
  ifs>>data;

  std::cout<<"Input data: "<<std::endl<<data<<std::endl;

  // Now process each video file
  std::vector<std::string> all_videos = data.all_videos();
  for (unsigned i=0;i<all_videos.size();++i)
    process_video(data.input_dir, data.output_dir,all_videos[i]);
}
