#include "mobio_parser.h"
#include <cctype>
#include <sstream>
#include <map>
#include <algorithm>

//: Read a block of text from a stream.
// This function will read through a stream, and store the text found to a string.
// The function terminates when it finds the closing brace.
//
// The stream's fail bit will be set on error. Comment lines beginning with //
// will be stripped.
// \param open_already should be true if the client has already
// read the opening brace. If set to false and the first non-whitespace character is
// not an opening brace then that character will be left in the stream and "{} will be returned.
// \return the text including the opening and closing braces.
// \param comment Lines beginning with white space followed by this string will be ignored.
std::string mobio_parse_block(std::istream &ifs, 
                           bool open_already /*= false*/, 
                           const char * comment /*= "//"*/)
{
  if (!ifs) return "{}";
  //: The last character to be read from the stream
  char c;
  if (!open_already)
  {
    if (ifs.eof())
      return "{}";
    ifs >> c;

    if (!ifs) return "{}";
    if (c != '{')
    {
      ifs.putback(c); // push c back into stream.
      return "{}";
    }
  }
  // The stored string.
  std::string s="{";
  // The current line is stored separately
  // before being added to s, in case it turns out to be a comment.
  std::string s2="";
  // The number of open braces.
  unsigned level=1;
  // The current position in a comment token.
  unsigned comment_position=0;
  // length of the comment token;
  const unsigned comment_length = std::strlen(comment);
  // true if we are currently in the whitespace at the beggining of a line
  bool newline=true;

  while (!(!ifs))
  {
    // read in one line at a time, to make decisions about comments
    while (!(!ifs))
    {
      ifs.get(c);
      s2 += c;
      if (c=='\n')
      {
        s+=s2;

        // reset variables to default values
        s2="";
        newline = true;
        comment_position = 0;
      }
      else if (std::isspace(c))
        // ignore leading white space
        comment_position = 0;
      else if (newline && comment_position < comment_length
               &&  c==comment[comment_position])
      {
        if (++comment_position == 2) // why 2? not comment_length?
        {
          // throw away rest of line
          std::string dummy;
          std::getline(ifs, dummy);

          // reset variables to default values
          s2 = "";
          newline = true; //false;
          comment_position = 0;
        }
      }
      else
      {
        newline = false;
        comment_position = 0;
        if (c=='{') ++level;
        else if (c=='}')
        {
          if (--level==0) // Found final closing brace
          {
            s+=s2;
            // check for empty space between braces
            for (unsigned i = 1; i+1 < s.size(); ++i)
              // if we find a non-space character anywhere then we're done so return s
              if (!std::isspace(s[i])) return s;

            // otherwise return empty braces
            return "{}";
          }
        }
      }
    }
  }
  std::cerr<<"Read problem (possibly end-of-file) before closing '}'"<<std::endl;
  ifs.clear(std::ios::badbit); // Set an unrecoverable IO error on stream
  return "{}";
}

//: Extracts list of strings from stream
//  Expects format "{ str1 str2 str3 ... }".
//  Aborts if fails to find opening or closing brace.
void mobio_parse_string_list(std::istream& is,
                             std::vector<std::string>& str_list)
{
  str_list.resize(0);

  char c;  // For first character
  std::string data_str; // For rest of string

  is>>c;
  if (c!='{')
  {
    std::cerr<<"mobio_parse_string_list Expected { at start."<<std::endl;
    abort();
  }

  is>>data_str;
  while (data_str!="}" && !is.eof())
  {
    str_list.push_back(data_str); // Append string to vector str_list
    is>>data_str>>std::ws; // Read string followed by whitespace
  }

  if (data_str!="}")
  {
    std::cerr<<"mobio_parse_string_list: "
            <<"Reached end of stream without finding closing }"<<std::endl;
    std::cerr<<"Last string was: "<<data_str<<std::endl;
    abort();
  }
}


// ============= mobio_indent ====================

// Get pointer to current indent for os
unsigned* indent_data(std::ostream& os)
{
  typedef std::map<void*, unsigned, std::less<void*> > maps2i_type;
  // Global record of tab information for streams.
  // Allows data to persist beyond the lifetime of the indent object itself,
  // which may be mercifully brief
  static maps2i_type indent_data_map;

  maps2i_type::iterator entry = indent_data_map.find(&os);
  if (entry==indent_data_map.end())
  {
    // Create a new entry
    indent_data_map[&os]=0;
    entry = indent_data_map.find(&os);
  }

  return &((*entry).second);
}

//: Writes a number of spaces depending on current indent for stream
std::ostream& operator<<(std::ostream& os, const mobio_indent&)
{
  unsigned n = *indent_data(os);
  for (unsigned i=0;i<n;++i)
    os<<' ';
  return os;
}

//: Increase indentation associated with given stream
void mobio_inc_indent(std::ostream& os)
{
  (*indent_data(os))+=2;
}

//: Decrease indentation associated with given stream
void mobio_dec_indent(std::ostream& os)
{
  if ((*indent_data(os))>=2)
    (*indent_data(os))-=2;
}


/*

// ============= mobio_videodata ====================

//: Parses stream for video parameters
// Expects format "filename: ..."
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_videodata& f)
{
  std::string key, data_str;

  // Set f to default values
  f = mobio_videodata();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="{")
    { }  // Ignore
    if (key=="filename:")
    { is>>f.filename; }
    else if (key=="frame_length:")
    { is>>data_str; f.frame_length=std::atof(data_str.c_str()); }
    else if (key=="start_frame:")
    { is>>data_str; f.start_frame=std::atoi(data_str.c_str()); }
    else if (key=="end_frame:")
    { is>>data_str; f.end_frame=std::atoi(data_str.c_str()); }
    else
    {
      std::cerr<<"mobio_videodata: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  return is;
}

//: Writes video parameters to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_videodata& f)
{
  os<<"filename: "<<f.filename<<std::endl;
  os<<"frame_length: "<<f.frame_length<<std::endl;
  os<<"start_frame: "<<f.start_frame<<std::endl;
  os<<"end_frame: "<<f.end_frame<<std::endl;

  return os;
}

*/

// ============= mobio_id ====================

//: Parses stream for ID
// Expects format "{ name: ... videos: ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_id& id)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"mobio_id: Expected { at start of id data."<<std::endl;
    abort();
  }

  // Set fb to default values
  id = mobio_id();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="name:")
    { is>>id.name; }
    else if (key=="videos:")
    { 
      data_str = mobio_parse_block(is);
      std::stringstream ss(data_str);
      mobio_parse_string_list(ss,id.videos);
    }
    else
    {
      std::cerr<<"mobio_id: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"mobio_id: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Parses stream for set of IDs
// Expects format "{ ID: { name: ... } ID: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_id>& vec_id)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"vec<mobio_id>: "
            <<" Expected { at start of sets of id data."<<std::endl;
    abort();
  }

  // Set vec_fb to default values
  vec_id.resize(0);

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="ID:")
    { 
      // cut out just the part of the string with the mobio_id parameters
      data_str=mobio_parse_block(is); 

      mobio_id id;
      std::stringstream ss(data_str);
      ss>>id;
      vec_id.push_back(id);
    }
    else
    {
      std::cerr<<"vec<mobio_id>: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"vec<mobio_id>: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Writes ID to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_id& id)
{
  os<<"ID: { name: "<<id.name;
  os<<" videos: { ";
  for (unsigned i=0;i<id.videos.size();++i)
    os<<id.videos[i]<<" ";
  os<<"} }";

  return os;
}

//: Writes set of IDs to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_id>& vec_id)
{
  os<<"enrol: ";
  if (vec_id.size()==0)
    os<<"{ }";
  else if (vec_id.size()==1)
    os<<"{ "<<vec_id[0]<<" }";
  else
  {
    os<<std::endl;
    os<<mobio_indent()<<"{"<<std::endl;
    mobio_inc_indent(os);
    for (unsigned i=0;i<vec_id.size();++i)
      os<<mobio_indent()<<vec_id[i]<<std::endl;
    mobio_dec_indent(os);
    os<<mobio_indent()<<"}";
  }

  return os;
}


// ============= mobio_test ====================

//: Parses stream for test
// Expects format "{ ID: { ... } challenge: { ... } }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_test& test)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"mobio_test: Expected { at start of test data."<<std::endl;
    abort();
  }

  // Set fb to default values
  test = mobio_test();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="ID:")
    { is>>test.actual_id; }

    else if (key=="challenge:")
    { 
      data_str = mobio_parse_block(is);
      std::stringstream ss(data_str);
      mobio_parse_string_list(ss,test.claimed_ids);
    }

    else
    {
      std::cerr<<"mobio_test: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"mobio_test: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Parses stream for set of test
// Expects format "{ test: { ... } test: { ... } }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_test>& vec_test)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"vec<mobio_test>: "
            <<" Expected { at start of sets of test data."<<std::endl;
    abort();
  }

  // Set vec_fb to default values
  vec_test.resize(0);

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="test:")
    { 
      // cut out just the part of the string with the mobio_id parameters
      data_str=mobio_parse_block(is); 

      mobio_test test;
      std::stringstream ss(data_str);
      ss>>test;
      vec_test.push_back(test);
    }

    else
    {
      std::cerr<<"vec<mobio_test>: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"vec<mobio_test>: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Writes test to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_test& test)
{
  os<<"test: { "<<test.actual_id;
  os<<" challenge: { ";
  for (unsigned i=0;i<test.claimed_ids.size();++i)
    os<<test.claimed_ids[i]<<" ";
  os<<"} }";

  return os;
}

//: Writes set of tests to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_test>& vec_test)
{
  os<<"tests: ";
  if (vec_test.size()==0)
    os<<"{ }";
  else if (vec_test.size()==1)
    os<<"{ "<<vec_test[0]<<" }";
  else
  {
    os<<std::endl;
    os<<mobio_indent()<<"{"<<std::endl;
    mobio_inc_indent(os);
    for (unsigned i=0;i<vec_test.size();++i)
      os<<mobio_indent()<<vec_test[i]<<std::endl;
    mobio_dec_indent(os);
    os<<mobio_indent()<<"}";
  }

  return os;
}


// ============= mobio_inputfile ====================

// Return list of all videos
std::vector<std::string> mobio_inputfile::all_videos()
{
  std::vector<std::string> all;

  // videos for enrolment
  for (unsigned i=0;i<ids.size();++i)
    for (unsigned j=0;j<ids[i].videos.size();++j) 
      all.push_back(ids[i].videos[j]);

  // videos for testing
  for (unsigned i=0;i<tests.size();++i)
    for (unsigned j=0;j<tests[i].actual_id.videos.size();++j) 
      all.push_back(tests[i].actual_id.videos[j]);

  return all;
}

// Return list of all enrolled IDs
std::vector<std::string> mobio_inputfile::all_ids()
{
  std::vector<std::string> all;

  for (unsigned i=0;i<ids.size();++i)
    all.push_back(ids[i].name);

  return all;
}

bool mobio_inputfile::valid_claimed_ids()
{
  // get vector of all enrolled IDs
  std::vector<std::string> ids = all_ids();

  for (unsigned int t=0; t<tests.size(); t++)
  {
    // if "ALL" is in tests.claimed_ids then replace with all_ids()
    std::vector<std::string>::iterator it;
    it = find(tests[t].claimed_ids.begin(),tests[t].claimed_ids.end(),"ALL");
    if ( it != tests[t].claimed_ids.end() ) // "ALL" found
      tests[t].claimed_ids = ids;

    for (unsigned int i=0; i<tests[t].claimed_ids.size(); i++)
      if ( find(ids.begin(),ids.end(),tests[t].claimed_ids[i]) == ids.end() )
      {
        std::cerr<<"ID "<<tests[t].claimed_ids[i]<<" not enrolled"<<std::endl;
        return false;
      }
  }

  return true;
}

//: Parses stream for input parameters (directories, IDs and datasets)
// Expects format "input_dir: ... "
// Reads to the end of the file, or to next unread closing brace "}"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        mobio_inputfile& f)
{
  std::string key, data_str;
  
  const int MAXLEN=255;
  char ignore_me[MAXLEN];

  // Set f to default values
  f = mobio_inputfile();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="{")
    { }  // Ignore

    // Comment line, so read to end
    if ( key.size()>1 && (key[0]=='/') && (key[1]=='/') )
    { is.getline(ignore_me, MAXLEN); }

    else if (key=="sparams_file:")
    { is>>f.sparams_file; }

    else if (key=="input_dir:")
    { 
      is>>f.input_dir; 
      if ( f.input_dir[f.input_dir.size()-1] != '/' &&
           f.input_dir[f.input_dir.size()-1] != '\\' )
        f.input_dir = f.input_dir+'/';
    }

    else if (key=="model_dir:")
    { 
      is>>f.model_dir; 
      if ( f.model_dir[f.model_dir.size()-1] != '/' &&
           f.model_dir[f.model_dir.size()-1] != '\\' )
        f.model_dir = f.model_dir+'/';
    }

    else if (key=="output_dir:")
    { 
      is>>f.output_dir; 
      if ( f.output_dir[f.output_dir.size()-1] != '/' &&
           f.output_dir[f.output_dir.size()-1] != '\\' )
        f.output_dir = f.output_dir+'/';
    }

    else if (key=="file_ext:")
    { is>>f.file_ext; }

    else if (key=="enrol:")
    {
      data_str = mobio_parse_block(is);
      std::stringstream ss(data_str);
      ss>>f.ids;
    }

    else if (key=="tests:")
    { 
      data_str = mobio_parse_block(is);
      std::stringstream ss(data_str);
      ss>>f.tests;
    }

    else
    {
      std::cerr<<"mobio_inputfile: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if ( !f.valid_claimed_ids() )
    abort();

  return is;
}

//: Writes parameters to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_inputfile& f)
{
  os<<mobio_indent()<<"input_dir: "<<f.input_dir<<std::endl;
  os<<mobio_indent()<<"model_dir: "<<f.model_dir<<std::endl;
  os<<mobio_indent()<<"output_dir: "<<f.output_dir<<std::endl;
  os<<mobio_indent()<<"file_ext: "<<f.file_ext<<std::endl;
  os<<mobio_indent()<<f.ids<<std::endl;
  os<<mobio_indent()<<f.tests<<std::endl;
  
  return os;
}


// ============= mobio_facebox ====================

//: Utility constructor
mobio_facebox::mobio_facebox(double cx1, double cy1, double wx1, double wy1,
                double score1)
  : cx(cx1),cy(cy1),wx(wx1),wy(wy1),ax(0),ay(0),az(0),score(score1)
{
}

//: Parses stream for facebox
// Expects format "{ cx: ... cy: ... wx: ... wy: ... score: ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_facebox& fb)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"mobio_facebox: Expected { at start of facebox data."<<std::endl;
    abort();
  }

  // Set fb to default values
  fb = mobio_facebox();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="cx:")
    { is>>data_str; fb.cx=std::atof(data_str.c_str()); }
    else if (key=="cy:")
    { is>>data_str; fb.cy=std::atof(data_str.c_str()); }
    else if (key=="wx:")
    { is>>data_str; fb.wx=std::atof(data_str.c_str()); }
    else if (key=="wy:")
    { is>>data_str; fb.wy=std::atof(data_str.c_str()); }
    else if (key=="score:")
    { is>>data_str; fb.score=std::atof(data_str.c_str()); }
    else if (key=="ax:")
    { is>>data_str; fb.ax=std::atof(data_str.c_str()); }
    else if (key=="ay:")
    { is>>data_str; fb.ay=std::atof(data_str.c_str()); }
    else if (key=="az:")
    { is>>data_str; fb.az=std::atof(data_str.c_str()); }
    else
    {
      std::cerr<<"mobio_facebox: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"mobio_facebox: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Parses stream for sets of faceboxes (one frame)
// Expects format "{ facebox: { cx: ... } facebox: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_facebox>& vec_fb)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"mobio_facebox: "
            <<" Expected { at start of sets of facebox data."<<std::endl;
    abort();
  }

  // Set vec_fb to default values
  vec_fb.resize(0);

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="facebox:")
    { 
      // cut out just the part of the string with the facebox parameters
      data_str=mobio_parse_block(is); 

      mobio_facebox fb;
      std::stringstream ss(data_str);
      ss>>fb;
      vec_fb.push_back(fb);
    }
    else
    {
      std::cerr<<"mobio_facebox: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"sets of mobio_facebox: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Parses stream for sets of sets of faceboxes (all frames)
// Expects format "{ frame: { facebox: ... } frame: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<std::vector<mobio_facebox> >& frames)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"mobio_facebox: Expected { at start of id data."<<std::endl;
    abort();
  }

  frames.resize(0);

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="frame:")
    { 
      data_str=mobio_parse_block(is); 
      std::vector<mobio_facebox> fb;
      std::stringstream ss(data_str);
      ss>>fb;
      frames.push_back(fb);
    }
    else
    {
      std::cerr<<"frames: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"sets of mobio_facebox: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Writes facebox to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_facebox& fb)
{
  os<<"facebox: {";
  os<<" cx: "<<fb.cx<<" cy: "<<fb.cy;
  os<<" wx: "<<fb.wx<<" wy: "<<fb.wy;
  if (fb.ax!=0.0) os<<" ax: "<<fb.ax;
  if (fb.ay!=0.0) os<<" ay: "<<fb.ay;
  if (fb.az!=0.0) os<<" az: "<<fb.az;
  os<<" score: "<<fb.score;
  os<<" }";

  return os;
}

//: Writes set of faceboxes to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_facebox>& vec_fb)
{
  os<<"frame: ";
  if (vec_fb.size()==0)
    os<<"{ }";
  else if (vec_fb.size()==1)
    os<<"{ "<<vec_fb[0]<<" }";
  else 
  {
    os<<std::endl;
    os<<mobio_indent()<<"{"<<std::endl;
    mobio_inc_indent(os);
    for (unsigned i=0;i<vec_fb.size();++i)
      os<<mobio_indent()<<vec_fb[i]<<std::endl;
    mobio_dec_indent(os);
    os<<mobio_indent()<<"}";
  }

  return os;
}

//: Writes set of set of faceboxes to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector< std::vector <mobio_facebox> >& frames)
{
  os<<"frames: ";
  if (frames.size()==0)
    os<<" { }";
  else if (frames.size()==1)
    os<<"{ "<<frames[0]<<" }";
  else
  {
    os<<std::endl;
    os<<mobio_indent()<<"{"<<std::endl;
    mobio_inc_indent(os);
    for (unsigned i=0;i<frames.size();++i)
      os<<mobio_indent()<<frames[i]<<std::endl;
    mobio_dec_indent(os);
    os<<mobio_indent()<<"}";
  }

  return os;
}

// ==================== mobio_video_faceboxes =================

//: Writes mobio_video_faceboxes to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_video_faceboxes& vfb)
{
  os<<mobio_indent()<<"filename: "<<vfb.filename<<std::endl;
  os<<mobio_indent()<<"frame_length: "<<vfb.frame_length<<std::endl;
  os<<mobio_indent()<<"start_frame: "<<vfb.start_frame<<std::endl;
  os<<mobio_indent()<<vfb.frame_data<<std::endl;

  return os;
}

//: Parses stream for mobio_video_faceboxes
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        mobio_video_faceboxes& vfb)
{
  std::string key,data_str;

  // Set to default values
  vfb = mobio_video_faceboxes();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="{")
    { }  // Ignore
    else if (key=="filename:")
    { is>>vfb.filename; }
    else if (key=="frame_length:")
    { is>>data_str; vfb.frame_length=std::atof(data_str.c_str()); }
    else if (key=="start_frame:")
    { is>>data_str; vfb.start_frame=std::atoi(data_str.c_str()); }
    else if (key=="frames:")
    {
      data_str = mobio_parse_block(is);
      std::stringstream ss(data_str);
      ss>>vfb.frame_data;
    }
    else
    {
      std::cerr<<"mobio_video_faceboxes: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  return is;

}


// ==================== mobio_facept ==========================

//: Parses stream for facept
// Expects format "{ x: ... y: ... score: ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        mobio_facept& pt)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"mobio_facept: Expected { at start of stream."<<std::endl;
    abort();
  }

  // Set p to default values
  pt = mobio_facept();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="x:")
    { is>>data_str; pt.x=std::atof(data_str.c_str()); }
    else if (key=="y:")
    { is>>data_str; pt.y=std::atof(data_str.c_str()); }
    else if (key=="score:")
    { is>>data_str; pt.score=std::atof(data_str.c_str()); }
    else
    {
      std::cerr<<"mobio_facept: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"mobio_facept: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Parses stream for set of facepts (one face)
// Expects format "{ p: { x: ... } p: { ... } p: { ... } }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_facept>& vec_pt)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"vec<mobio_facept>: "
            <<"{ expected, "<<c<<" found"<<std::endl;
    abort();
  }

  vec_pt.resize(0);

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="pt:")
    { 
      data_str=mobio_parse_block(is);
      mobio_facept pt;
      std::stringstream ss(data_str);
      ss>>pt;
      vec_pt.push_back(pt);
    }
    else
    {
      std::cerr<<"vec<mobio_facept>: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"vec<mobio_facept>: "
            <<"Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Writes facept to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_facept& pt)
{
  os<<"pt: { x: "<<pt.x<<" y: "<<pt.y<<" score: "<<pt.score<<" }";

  return os;
}

//: Writes set of facepts to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_facept>& vec_pt)
{
  os<<"points: ";
  if (vec_pt.size()==0)
    os<<"{ }";
  else if (vec_pt.size()==1)
    os<<"{ "<<vec_pt[0]<<" }";
  else
  {
    os<<std::endl;
    os<<mobio_indent()<<"{"<<std::endl;
    mobio_inc_indent(os);
    for (unsigned i=0;i<vec_pt.size();++i)
      os<<mobio_indent()<<vec_pt[i]<<std::endl;
    mobio_dec_indent(os);
    os<<mobio_indent()<<"}";
  }

  return os;
}


// ==================== mobio_face_points ==========================

std::vector<double> mobio_face_points::eye_centres()
{
  // return vector = < left_x, left_y, right_x, right_y >

  std::vector<double> centres(4,0.0);

  if ( points.size()==68 || points.size()==142)
  {
    // common 68pt/142pt configurations
    centres.push_back( points[36].x );
    centres.push_back( points[36].y );
    centres.push_back( points[31].x );
    centres.push_back( points[31].y );
  }

  else
  {
    // e.g. 20pt form used for BioID
    centres.push_back( points[1].x );
    centres.push_back( points[1].y );
    centres.push_back( points[0].x );
    centres.push_back( points[0].y );
  }

  return centres;
}

//: Parses stream for face
// Expects format "{ score: 0.87 pts: { x: ... } }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        mobio_face_points& face)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"mobio_face_points: { expected, "<<c<<" found."<<std::endl;
    abort();
  }

  // Set face to default values
  face = mobio_face_points();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="score:")
    { is>>data_str; face.score=std::atof(data_str.c_str()); }
    else if (key=="points:")
    { 
      // cut out just the part of the string with the point parameters
      data_str=mobio_parse_block(is);
      std::stringstream ss(data_str);
      ss>>face.points;
    }
    else
    {
      std::cerr<<"mobio_face_points: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"mobio_face_points: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Parses data for set of faces (one frame)
// Expects format "{ face: { score: ... } face: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_face_points>& vec_face)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"vec<mobio_face_points>: "
            <<" Expected { at start of sets of frame data."<<std::endl;
    abort();
  }

  vec_face.resize(0);

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="face:")
    { 
      data_str=mobio_parse_block(is);

      mobio_face_points face;
      std::stringstream ss(data_str);
      ss>>face;
      vec_face.push_back(face);
    }
    else
    {
      std::cerr<<"vec<mobio_face_points>: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"vec<mobio_face_points>: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Parses stream for set of set of faces (all frames)
// Expects format "{ frame: { face: { ... } frame: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector< std::vector<mobio_face_points> >& frames)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"vec<vec<mobio_face>>: Expected { at start of id data."<<std::endl;
    abort();
  }

  frames.resize(0);

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="frame:")
    { 
      data_str=mobio_parse_block(is);

      std::vector<mobio_face_points> vec_face;
      std::stringstream ss(data_str);
      ss>>vec_face;
      frames.push_back(vec_face);
    }
    else
    {
      std::cerr<<"vec<vec<mobio_face>>: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"vec<vec<mobio_face>>: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Writes face to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_face_points& face)
{
  os<<"face: "<<std::endl;
  os<<mobio_indent()<<"{"<<std::endl;
  mobio_inc_indent(os);
  os<<mobio_indent()<<"score: "<<face.score<<std::endl;
  os<<mobio_indent()<<face.points<<std::endl;
  mobio_dec_indent(os);
  os<<mobio_indent()<<"}";

  return os;
}

//: Writes set of faces to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_face_points>& vec_face)
{
  os<<"frame: ";
  if (vec_face.size()==0)
    os<<"{ }";
  else
  {
    os<<std::endl;
    os<<mobio_indent()<<"{"<<std::endl;
    mobio_inc_indent(os);
    for (unsigned i=0;i<vec_face.size();++i)
      os<<mobio_indent()<<vec_face[i]<<std::endl;
    mobio_dec_indent(os);
    os<<mobio_indent()<<"}";
  }

  return os;
}

//: Writes set of set of faces to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector< std::vector<mobio_face_points> >& frames)
{
  os<<"frames: ";
  if (frames.size()==0)
    os<<"{ }";
  else if (frames.size()==1)
    os<<"{ "<<frames[0]<<" }";
  else 
  {
    os<<std::endl;
    os<<mobio_indent()<<"{"<<std::endl;
    mobio_inc_indent(os);
    for (unsigned i=0;i<frames.size();++i)
      os<<mobio_indent()<<frames[i]<<std::endl;
    mobio_dec_indent(os);
    os<<mobio_indent()<<"}";
  }

  return os;
}

// ==================== mobio_video_face_points =================

//: Writes mobio_video_face_points to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_video_face_points& vfp)
{
  os<<mobio_indent()<<"filename: "<<vfp.filename<<std::endl;
  os<<mobio_indent()<<"frame_length: "<<vfp.frame_length<<std::endl;
  os<<mobio_indent()<<"start_frame: "<<vfp.start_frame<<std::endl;
  os<<mobio_indent()<<vfp.frame_data<<std::endl;

  return os;
}

//: Parses stream for mobio_video_face_points
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        mobio_video_face_points& vfp)
{
  std::string key,data_str;

  // Set to default values
  vfp = mobio_video_face_points();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="{")
    { }  // Ignore
    else if (key=="filename:")
    { is>>vfp.filename; }
    else if (key=="frame_length:")
    { is>>data_str; vfp.frame_length=std::atof(data_str.c_str()); }
    else if (key=="start_frame:")
    { is>>data_str; vfp.start_frame=std::atoi(data_str.c_str()); }
    else if (key=="frames:")
    {
      data_str = mobio_parse_block(is);
      std::stringstream ss(data_str);
      ss>>vfp.frame_data;
    }
    else
    {
      std::cerr<<"mobio_video_face_points: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  return is;

}


// ==================== mobio_segment =================

//: Parses stream for segment
// Expects format "{ score: ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_segment& segment)
{
  std::string key;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"mobio_segment: Expected { at start of segment data."<<std::endl;
    abort();
  }

  // Set to default values
  segment = mobio_segment();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="score:")
    { is>>segment.score; }

    else
    {
      std::cerr<<"mobio_segment: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"mobio_segment: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Parses stream for set of segments
// Expects format "{ segment: { score: ... } segment: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_segment>& vec_segment)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"vec<mobio_segment>: "
            <<" Expected { at start of sets of segment data."<<std::endl;
    abort();
  }

  // Set vec_fb to default values
  vec_segment.resize(0);

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="segment:")
    { 
      // cut out just the part of the string with the mobio_id parameters
      data_str=mobio_parse_block(is); 

      mobio_segment segment;
      std::stringstream ss(data_str);
      ss>>segment;
      vec_segment.push_back(segment);
    }

    else
    {
      std::cerr<<"vec<mobio_segment>: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"vec<mobio_segment>: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Writes segment to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_segment& segment)
{
  os<<"segment: { score: "<<segment.score<<" }";

  return os;
}

//: Writes set of segments to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_segment>& vec_segment)
{
  os<<"segments: ";
  if (vec_segment.size()==0)
    os<<"{ }";
  else if (vec_segment.size()==1)
    os<<"{ "<<vec_segment[0]<<" }";
  else
  {
    os<<std::endl;
    os<<mobio_indent()<<"{"<<std::endl;
    mobio_inc_indent(os);
    for (unsigned i=0;i<vec_segment.size();++i)
      os<<mobio_indent()<<vec_segment[i]<<std::endl;
    mobio_dec_indent(os);
    os<<mobio_indent()<<"}";
  }

  return os;
}


// ==================== mobio_result =================

//: Parses stream for result
// Expects format "{ claimed_id: ... total_score: ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_result& result)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"mobio_result: Expected { at start of result data."<<std::endl;
    abort();
  }

  // Set to default values
  result = mobio_result();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="claimed_id:")
    { is>>result.claimed_id; }

    else if (key=="total_score:")
    { is>>result.total_score; }

    else if (key=="segment_length:")
    { is>>result.segment_length; }

    else if (key=="start_time:")
    { is>>result.start_time; }

    else if (key=="segments:")
    {
      data_str = mobio_parse_block(is);
      std::stringstream ss(data_str);
      ss>>result.scores;
    }

    else
    {
      std::cerr<<"mobio_result: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"mobio_result: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Parses stream for set of results
// Expects format "{ result: { claimed_id: ... } result: { ... } ... }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is, 
                        std::vector<mobio_result>& vec_result)
{
  std::string key,data_str;
  char c;  // For first character
  is>>c;
  if (c!='{')
  {
    std::cerr<<"vec<mobio_result>: "
            <<" Expected { at start of sets of result data."<<std::endl;
    abort();
  }

  // Set vec_fb to default values
  vec_result.resize(0);

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="result:")
    { 
      // cut out just the part of the string with the mobio_id parameters
      data_str=mobio_parse_block(is); 

      mobio_result result;
      std::stringstream ss(data_str);
      ss>>result;
      vec_result.push_back(result);
    }
    else
    {
      std::cerr<<"vec<mobio_result>: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  if (key!="}")
  {
    std::cerr<<"vec<mobio_result>: Reached end of stream without finding closing }"<<std::endl;
    abort();
  }

  return is;
}

//: Writes result to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_result& result)
{
  os<<"result:"<<std::endl
    <<mobio_indent()<<"{"<<std::endl;
  mobio_inc_indent(os);
  os<<mobio_indent()<<"claimed_id: "<<result.claimed_id<<std::endl
    <<mobio_indent()<<"total_score: "<<result.total_score<<std::endl
    <<mobio_indent()<<"segment_length: "<<result.segment_length<<std::endl
    <<mobio_indent()<<"start_time: "<<result.start_time<<std::endl
    <<mobio_indent()<<result.scores<<std::endl;
  mobio_dec_indent(os);
  os<<mobio_indent()<<"}";

  return os;
}

//: Writes set of ID_score's to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const std::vector<mobio_result>& vec_result)
{
  os<<"results: ";
  if (vec_result.size()==0)
    os<<"{ }";
  else
  {
    os<<std::endl;
    os<<mobio_indent()<<"{"<<std::endl;
    mobio_inc_indent(os);
    for (unsigned i=0;i<vec_result.size();++i)
      os<<mobio_indent()<<vec_result[i]<<std::endl;
    mobio_dec_indent(os);
    os<<mobio_indent()<<"}";
  }

  return os;
}


// ==================== mobio_outputfile =================

//: Parses stream for output file
// Expects format "{ filename: ... actual_id: ... results: { ... } }"
// Aborts with an error if parsing fails
std::istream& operator>>(std::istream& is,
                        mobio_outputfile& outputfile)
{
  std::string key,data_str;

  // Set to default values
  outputfile = mobio_outputfile();

  is>>key;
  while (key!="}" && !is.eof())
  {
    if (key=="filename:")
    { is>>outputfile.filename; }

    else if (key=="actual_id:")
    { is>>outputfile.actual_id; }

    else if (key=="results:")
    {
      data_str = mobio_parse_block(is);
      std::stringstream ss(data_str);
      ss>>outputfile.results;
    }

    else
    {
      std::cerr<<"mobio_outputfile: Unknown key: <"<<key<<">"<<std::endl;
      abort();
    }
    // Read the next key string
    is>>key;
  }

  return is;
}

//: Writes output file to stream in easy to parse format
std::ostream& operator<<(std::ostream& os, 
                        const mobio_outputfile& outputfile)
{
  os<<"filename: "<<outputfile.filename<<std::endl
    <<"actual_id: "<<outputfile.actual_id<<std::endl
    <<outputfile.results
    <<std::endl;

  return os;
}

