// Example of face detector tool, demonstrating use of interfaces

#include <fstream>           // Standard file stream classes
#include "mobio_parser.h"    // mobio data classes + parsing code

// Strip the final ".xxx" from s
std::string strip_extension(const std::string& s)
{
  size_t p = s.find_last_of(".");
  if (p==std::string::npos) return s;  // No "." found
  return s.substr(0,p);
}

// Reads in video from video_name in video_dir
// Processes it to find faces
// Writes output to file in output_dir
void process_video(const std::string& video_dir,
                   const std::string& output_dir,
                   const std::string& video_name)
{
  std::string video_path = video_dir+video_name;
  std::cout<<"Processing video "<<video_path<<std::endl;

  // Load video from video_path
  // ...

  unsigned n_frames=5;  // Set to number of frames in video
  
  // Set up structure to hold results
  mobio_video_faceboxes results;
  results.filename = video_name;
  results.frame_length = 40; // Or whatever
  results.start_frame = 0;

  for (unsigned i=results.start_frame;i<n_frames;++i)
  {
    std::vector<mobio_facebox> face_boxes;
    // Process frame i, creating one element in face_boxes for each
    // found face.
    // ...
    // For example:
    face_boxes.push_back(mobio_facebox(10,15,  20,30,  0.65));

    // Store result for this frame
    results.frame_data.push_back(face_boxes);
  }

  // Save the results to output directory
  std::string out_path = output_dir+strip_extension(video_name)
                          +"_faceboxes.txt";

  std::ofstream ofs(out_path.c_str());
  if (!ofs)
  {
    std::cerr<<"Unable to open file "<<out_path<<" for output."<<std::endl;
    abort();
  }
  ofs<<results;
  ofs.close();
}

void print_usage()
{
  std::cout<<"Example of face detector tool, demonstrating use of interfaces"<<std::endl;
}


int main(int argc, char* argv[])
{
  if (argc<3)
  {
    print_usage();
    return 0;
  }

  // Process the command line
  std::string input_path(argv[2]);

  // Read in parameters by parsing input text file
  mobio_inputfile data;
  std::ifstream ifs(input_path.c_str());
  ifs>>data;

  std::cout<<"Input data: "<<std::endl<<data<<std::endl;

  // Now process each video file
  std::vector<std::string> all_videos = data.all_videos();
  for (unsigned i=0;i<all_videos.size();++i)
    process_video(data.input_dir, data.output_dir,all_videos[i]);
}
