/*
Copyright (c) 2011 Idiap Research Institute, http://www.idiap.ch/
Written by Carl Scheffler <carl.scheffler@gmail.com>

This file is part of FaceColorModel.

FaceColorModel is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License version 3 as
published by the Free Software Foundation.

FaceColorModel is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with FaceColorModel. If not, see <http://www.gnu.org/licenses/>.
*/
#include "FaceColorModelWrapper.h"

#include <stdio.h>
#include <stdexcept>

FaceColorModelWrapper::FaceColorModelWrapper(double iKMrf/*= 2*/,
					     int iSkinType/*=FCM_CONTINUOUS*/,
					     int iHairType/*=FCM_CONTINUOUS*/) {
  import_array();
  mpFaceColorModel =
    new FaceColorModel::FaceColorModel(iKMrf, iSkinType, iHairType);
}

FaceColorModelWrapper::~FaceColorModelWrapper() {
  delete mpFaceColorModel;
}

void FaceColorModelWrapper::adapt_to(CvMat const* ipImageMat) {
  // Convert input to IplImage format
  IplImage imageHeader;
  IplImage* pImage = cvGetImage(ipImageMat, &imageHeader);
  mpFaceColorModel->adapt_to(pImage);
}

PyObject* FaceColorModelWrapper::channel_log_likelihood(
  CvMat const* ipImageMat) {

  // Convert input to IplImage format
  IplImage imageHeader;
  IplImage* pImage = cvGetImage(ipImageMat, &imageHeader);

  // Create Python array for output
  int dim[] = {FaceColorModel::FCM_NUM_CHANNELS, pImage->height, pImage->width};
  PyObject* opResult = PyArray_FromDims(3, dim, PyArray_DOUBLE);

  mpFaceColorModel->channel_log_likelihood(
    pImage, (double*)((PyArrayObject*)opResult)->data);

  return opResult;
}

void FaceColorModelWrapper::reset_to_prior() {
  mpFaceColorModel->reset_to_prior();
}

PyObject* FaceColorModelWrapper::get_color_posterior(const char* ipChannel) {
  int channelIndex;
  std::string iChannel(ipChannel);
  if(iChannel == "skin")
    channelIndex = FaceColorModel::FCM_CHANNEL_SKIN;
  else if(iChannel == "hair")
    channelIndex = FaceColorModel::FCM_CHANNEL_HAIR;
  else if(iChannel == "clothes")
    channelIndex = FaceColorModel::FCM_CHANNEL_CLOTHES;
  else if(iChannel == "background")
    channelIndex = FaceColorModel::FCM_CHANNEL_BACKGROUND;
  else
    throw std::runtime_error("Invalid channel: " + iChannel);

  int dim [1];
  if(mpFaceColorModel->get_channel_type(channelIndex) == FaceColorModel::FCM_DISCRETE)
    dim[0] = FaceColorModel::FCM_HISTOGRAM_SIZE;
  else
    dim[0] = 12;

  PyObject* opResult = PyArray_FromDims(1, dim, PyArray_DOUBLE);
  memcpy(((PyArrayObject*)opResult)->data, mpFaceColorModel->get_color_posterior(channelIndex), sizeof(double) * dim[0]);

  return opResult;
}

PyObject* FaceColorModelWrapper::get_class_posterior() {
  int dim [] = {FaceColorModel::FCM_NUM_CHANNELS,
		FaceColorModel::FCM_SCALED_SIZE,
		FaceColorModel::FCM_SCALED_SIZE};
  PyObject* opResult = PyArray_FromDims(3, dim, PyArray_DOUBLE);
  memcpy(((PyArrayObject*)opResult)->data,
	 mpFaceColorModel->get_class_posterior(),
	 sizeof(double) * dim[0] * dim[1] * dim[2]);

  return opResult;
}

int FaceColorModelWrapper::get_channel_type(int iChannelIndex) {
  return mpFaceColorModel->get_channel_type(iChannelIndex);
}

