//------------------------------------------------------------------------------
// Tasting families of features for image classification.
// 
// Copyright (c) 2011 Idiap Research Institute, http://www.idiap.ch/
// Written by Charles Dubout <charles.dubout@idiap.ch>
// 
// This file is part of Tasting.
// 
// Tasting is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
// 
// Tasting is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Tasting. If not, see <http://www.gnu.org/licenses/>.
//------------------------------------------------------------------------------

#include "InputSet.h"

#include <algorithm>
#include <cassert>

using namespace ML;

InputSet::InputSet()
: sampleStack_(), featureStack_(), matrixCache_(sampleStack_, featureStack_),
  labelCache_(sampleStack_), weightCache_(sampleStack_),
  heuristicCache_(featureStack_), imageCache_(sampleStack_),
  coordinatesCache_(sampleStack_) {
	// Nothing to do
}

InputSet::~InputSet() {
	// Nothing to do
}

unsigned int InputSet::nbSamples() const {
	assert(!sampleStack_.empty());
	return sampleStack_.back().size();
}

unsigned int InputSet::nbFeatures() const {
	assert(!featureStack_.empty());
	return featureStack_.back().size();
}

unsigned int InputSet::nbLabels() const {
	assert(!nbLabelStack_.empty());
	return nbLabelStack_.back();
}

unsigned int InputSet::nbImages() const {
	return nbImages_;
}

unsigned int InputSet::nbHeuristics() const {
	return nbHeuristics_;
}

void InputSet::pushSamples(const std::vector<unsigned int>& indices) {
	// There must be indices
	assert(!indices.empty());

	// Add the vector on the sample stack
	sampleStack_.push_back(indices);

	// The current number of labels is equal to the previous one
	nbLabelStack_.push_back(nbLabelStack_.back());
}

void InputSet::popSamples() {
	// There must be a pushed stack
	assert(sampleStack_.size() > 1);

	sampleStack_.pop_back();
	nbLabelStack_.pop_back();

	matrixCache_.empty();
	labelCache_.empty();
	weightCache_.empty();
	imageCache_.empty();
	coordinatesCache_.empty();
}

void InputSet::pushFeatures(const std::vector<unsigned int>& indices) {
	// There must be indices
	assert(!indices.empty());

	// Add the vector on the feature stack
	featureStack_.push_back(indices);
}

void InputSet::popFeatures() {
	// There must be a pushed stack
	assert(featureStack_.size() > 1);

	featureStack_.pop_back();

	matrixCache_.empty();
	heuristicCache_.empty();
}

const scalar_t* InputSet::features(unsigned int sample) const {
	// If there is no cache
	if(matrixCache_.empty()) {
		std::vector<scalar_t> data;
		matrixCache(data);
		matrixCache_.swap(data);
	}

	return matrixCache_.data() + sample * nbFeatures();
}

const scalar_t* InputSet::samples(unsigned int feature) const {
	// If there is no cache
	if(matrixCache_.empty()) {
		std::vector<scalar_t> data;
		matrixCache(data);
		matrixCache_.swap(data);
	}

	return matrixCache_.dataT() + feature * nbSamples();
}

const unsigned int* InputSet::labels() const {
	// If there is no cache
	if(labelCache_.empty()) {
		std::vector<unsigned int> data;
		labelCache(data);
		labelCache_.swap(data);
	}

	return labelCache_.data();
}

const scalar_t* InputSet::weights() const {
	// If there is no cache
	if(weightCache_.empty()) {
		std::vector<scalar_t> data;
		weightCache(data);
		weightCache_.swap(data);
	}

	return weightCache_.data();
}

const unsigned int* InputSet::images() const {
	// If there is no cache
	if(imageCache_.empty()) {
		std::vector<unsigned int> data;
		imageCache(data);
		imageCache_.swap(data);
	}

	return imageCache_.data();
}

const coordinates_t* InputSet::coordinates() const {
	// If there is no cache
	if(coordinatesCache_.empty()) {
		std::vector<coordinates_t> data;
		coordinatesCache(data);
		coordinatesCache_.swap(data);
	}

	return coordinatesCache_.data();
}

const unsigned int* InputSet::heuristics() const {
	// If there is no cache
	if(heuristicCache_.empty()) {
		std::vector<unsigned int> data;
		heuristicCache(data);
		heuristicCache_.swap(data);
	}

	return heuristicCache_.data();
}

void InputSet::swapFeatures(std::vector<scalar_t>& data, bool flag) {
	// Compute the current matrix of features if not done already
	if(flag) {
		features();
	}

	// Swap the matrix cache
	matrixCache_.swap(data, flag);
}

void InputSet::swapSamples(std::vector<scalar_t>& data, bool flag) {
	// Compute the current matrix of features if not done already
	if(flag) {
		samples();
	}

	// Swap the matrix cache
	matrixCache_.swapT(data, flag);
}

void InputSet::swapLabels(std::vector<unsigned int>& data, bool flag) {
	// Compute the current vector of labels if not done already
	if(flag) {
		labels();
	}

	// Set the new number of labels
	if(!data.empty()) {
		nbLabelStack_.back() = *std::max_element(data.begin(), data.end()) + 1;
	}
	else {
		nbLabelStack_.back() = *(nbLabelStack_.end() - 2);
	}

	// Swap the label cache
	labelCache_.swap(data);
}

void InputSet::swapWeights(std::vector<scalar_t>& data, bool flag) {
	// Compute the current vector of weights if not done already
	if(flag) {
		weights();
	}

	// Swap the weight cache
	weightCache_.swap(data);
}

void InputSet::swapImages(std::vector<unsigned int>& data, bool flag) {
	// Compute the current vector of images if not done already
	if(flag) {
		images();
	}

	// Clear the matrix cache
	if(!data.empty()) {
		std::vector<scalar_t> empty;
		matrixCache_.swap(empty);
	}

	// Swap the image cache
	imageCache_.swap(data);
}

void InputSet::swapCoordinates(std::vector<coordinates_t>& data, bool flag) {
	// Compute the current vector of weights if not done already
	if(flag) {
		coordinates();
	}

	// Clear the matrix cache
	if(!data.empty()) {
		std::vector<scalar_t> empty;
		matrixCache_.swap(empty);
	}

	// Swap the coordinates cache
	coordinatesCache_.swap(data);
}

void InputSet::swapHeuristics(std::vector<unsigned int>& data, bool flag) {
	// Compute the current vector of heuristics if not done already
	if(flag) {
		heuristics();
	}

	// Swap the heuristic cache
	heuristicCache_.swap(data);
}

void InputSet::clear() {
	// Clear all the caches
	matrixCache_.clear();
	labelCache_.clear();
	weightCache_.clear();
	imageCache_.clear();
	coordinatesCache_.clear();
	heuristicCache_.clear();

	// Reset the number of labels
	for(unsigned int i = 1; i < nbLabelStack_.size(); ++i) {
		nbLabelStack_[i] = nbLabelStack_[0];
	}
}
