% Copyright (c) 2016 Idiap Research Institute, http://www.idiap.ch/
% Written by Edgar Roman-Rangel, et al
% 
% This file is part of HOOSC implementation.
% 
% HOOSC is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License version 3 as
% published by the Free Software Foundation.
% 
% Foobar is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with Foobar. If not, see <http://www.gnu.org/licenses/>.

function HOOSC = hoosc(X, Y, T, x, y, t)
%- Histogram-of-Orientations Shape-Context desriptor, --------------------
%  [Roman-Rangel, et al 2011]
%
% Inputs
%       X and Y: are the column vectors containing the (x, y) coordinates
%           of the pivot points for with the descriptors will be computed.
%       T: contains the local orientations of the pivot points.
%       scales: contains the characteristic scales of the pivot points,
%           which might or not correspond to the average pairwise distance,
%           as in [Belongie et al., 2002].
%       x, y, and t: are the superset of all the points in the contour.
%
% Outputs
%       HOOSC: is a matrix whose rows are the HOOSC descriptors, one per pivot.
%
% ------------------------------------------------------------------------

global option;

%% Parameters
numOfThetaBins = 8;		% Works as good as 12
numOfPoints = numel(x);
numOfPivots = numel(X);


%% Polar coordinates
theta = zeros(numOfPivots, numOfPoints);
rho = zeros(numOfPivots, numOfPoints);
for pivot = 1 : numOfPivots
    dX = X(pivot) - x;
    dY = Y(pivot) - y;
    rho(pivot, :) = sqrt(dX.^2 + dY.^2);
    theta(pivot, :) = atan2(dY, dX);
end


theta = rem(rem(theta, 2 * pi) + 2 * pi, 2 * pi);


%% Make rho scale invariant
rho = rho ./ mean(pdist([x, y]));


%% Compute polar-grid Bins
rhoBin = ones(numOfPivots, numOfPoints);
for rbin = 1./[option.NumberRing:-1:1]%[0.5, 1.0]
    rhoBin = rhoBin + (rho > (option.ringsize*rbin));
end
thetaBin = 1 + floor(theta / (2 * pi / numOfThetaBins));


%% Prepare the Orientation kernel
H = orientationKernel(180, 8, 10);


%% HOOSC computation
if option.pos
    HOOSC = zeros(numOfPivots, option.NumberRing * 8 * numOfThetaBins+2);
else
    HOOSC = zeros(numOfPivots, option.NumberRing * 8 * numOfThetaBins);
end
for pivot = 1 : numOfPivots
    % Adjust local orientations according to pivot
    tlocal = mod(t - T(pivot), pi);
    tD = round(tlocal * 180 / pi);
    tD(tD == 0) = 180;
    
    % HOOSC computation
    if option.pos
        HOOSC_tmp = hooscCore(thetaBin(pivot, :), rhoBin(pivot, :), tD, H);
        HOOSC(pivot, :) = [X(pivot)/max(x), Y(pivot)/max(y), HOOSC_tmp]; clear HOOSC_tmp
    else
        HOOSC(pivot, :) = hooscCore(thetaBin(pivot, :), rhoBin(pivot, :), tD, H);
    end
end


end


%% HOOSC core computation
function HOOSC = hooscCore(theta, rho, T, H)

global option;

HOOSC = [];
for ring = 1 : option.NumberRing
    HOOSC = [HOOSC, hooscPerRing(theta(rho == ring), T(rho == ring), H)];
end

end


%% Compute HOOSC per ring
function HOOSC = hooscPerRing(theta, T, H)

% Find the points in the regions and integrate their Gaussian kernels
ring = zeros(8, 8);
for slice = 1 : 8
    ring(:, slice) = sum(H(T(theta == slice), :), 1)';
end
HOOSC = ring(:)';

% Normalize it
sumHOOSC = sum(HOOSC);
if (sumHOOSC > 0)
    HOOSC = HOOSC / sumHOOSC;
end

end
