//------------------------------------------------------------------------------
// Tasting families of features for image classification.
// 
// Copyright (c) 2011 Idiap Research Institute, http://www.idiap.ch/
// Written by Charles Dubout <charles.dubout@idiap.ch>
// 
// This file is part of Tasting.
// 
// Tasting is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
// 
// Tasting is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Tasting. If not, see <http://www.gnu.org/licenses/>.
//------------------------------------------------------------------------------

#include <algorithm>
#include <cassert>
#include <fstream>
#include <iostream>
#include <iterator>
#include <string>
#include <vector>

using namespace std;

int main(int argc, char* const argv[]) {
	// Check for correct usage of the command line
	if (argc != 3) {
		cerr << "Usage: " << argv[0] << " features.raw heuristics.txt" << endl;
		return 1;
	}

	// Try to open all files
	ifstream featureFile;
	ifstream heuristicFile;
	ofstream outputFile;

	// Enable exceptions
	featureFile.exceptions(ifstream::badbit);
	heuristicFile.exceptions(ifstream::badbit);
	outputFile.exceptions(ifstream::badbit);

	// Open the files
	featureFile.open(argv[1], ios::binary);
	heuristicFile.open(argv[2]);
	outputFile.open("tmp.raw", ios::binary);

	// Read the heuristic dimensions
	vector<size_t> heuristics;
	copy(istream_iterator<size_t>(heuristicFile), istream_iterator<size_t>(), back_inserter(heuristics));

	size_t nbFeatures = 0;

	for (int i = 0; i < heuristics.size(); ++i)
		nbFeatures += heuristics[i];

	// Compute the length of the feature file
	featureFile.seekg(0, ios::end);
	const size_t length = featureFile.tellg();
	featureFile.seekg(0, ios::beg);

	const size_t nbSamples = length / (nbFeatures * sizeof(float));

	// Two 512MB caches
	size_t nbCachedFeatures = min((1 << 29) / (nbSamples * sizeof(float)), nbFeatures);

	vector<float> cache(nbSamples * nbCachedFeatures);
	vector<float> transposedCache(nbSamples * nbCachedFeatures);

	cout << "#samples: " << nbSamples << endl;
	cout << "#features: " << nbFeatures << endl;
	cout << "#features/cache: " << nbCachedFeatures << endl;

	// Iterate over the output file
	for (size_t i = 0; i < nbFeatures; i += nbCachedFeatures) {
		nbCachedFeatures = min(nbCachedFeatures, nbFeatures - i);

		cout << "Reading features " << i << " to " << (i + nbCachedFeatures - 1) << endl;

		for (size_t j = 0; j < nbSamples; ++j) {
			if ((j % 100) == 0)
				cout << j << ' ' << flush;
			featureFile.seekg((j * nbFeatures + i) * sizeof(float), ios_base::beg);
			assert(featureFile.read(reinterpret_cast<char*>(&cache[j * nbCachedFeatures]),
				nbCachedFeatures * sizeof(float)));
		}

		cout << "\nTransposing the " << nbSamples << 'x' << nbCachedFeatures << " matrix." << endl;

		for (size_t y = 0; y < nbSamples; ++y)
			for (size_t x = 0; x < nbCachedFeatures; ++x)
				transposedCache[x * nbSamples + y] = cache[y * nbCachedFeatures + x];

		cout << "Writing the " << ((nbSamples * nbCachedFeatures * sizeof(float)) >> 20) << "MB matrix to file" << endl;
		assert(outputFile.write(reinterpret_cast<char*>(&transposedCache[0]), nbSamples * nbCachedFeatures * sizeof(float)));
		outputFile.flush();
	}
	
	char str[1024];
	sprintf(str, "mv tmp.raw %s", argv[1]);
	system(str);
}
