//------------------------------------------------------------------------------
// Simple example of a heuristic: Hue.
// Compute the hue of the pixels of the ROI.
// 
// Copyright (c) 2011 Wenqi You <wenqi.you@epfl.ch>
// 
// This file is part of Heuristics.
// 
// Heuristics is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
// 
// Heuristics is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Heuristics. If not, see <http://www.gnu.org/licenses/>.
//------------------------------------------------------------------------------

#include <mash/heuristic.h>
#include <algorithm>

using namespace Mash;


//------------------------------------------------------------------------------
/// The 'Hue' heuristic class
//------------------------------------------------------------------------------
class HueHeuristic: public Heuristic
{
    //_____ Implementation of Heuristic __________
public:
    virtual unsigned int dim();

    virtual scalar_t computeFeature(unsigned int feature_index);
};


//------------------------------------------------------------------------------
/// Creation function of the heuristic
//------------------------------------------------------------------------------
extern "C" Heuristic* new_heuristic()
{
    return new HueHeuristic();
}


/************************* IMPLEMENTATION OF Heuristic ************************/

unsigned int HueHeuristic::dim()
{
    // We have has many features than pixels in the region of interest
    unsigned int roi_size = roi_extent * 2 + 1;
    return roi_size * roi_size;
}


scalar_t HueHeuristic::computeFeature(unsigned int feature_index)
{
    // Compute the coordinates of the top-left pixel of the region of interest
    unsigned int x0 = coordinates.x - roi_extent;
    unsigned int y0 = coordinates.y - roi_extent;

    // Compute the coordinates of the pixel corresponding to the feature, in
    // the region of interest
    unsigned int roi_size = roi_extent * 2 + 1;
    unsigned int x = feature_index % roi_size;
    unsigned int y = (feature_index - x) / roi_size;

    // Return the pixel value corresponding to the desired feature
    RGBPixel_t** pLines = image->rgbLines();
	
	scalar_t R = pLines[y0 + y][x0 + x].r;
	scalar_t G = pLines[y0 + y][x0 + x].g;
	scalar_t B = pLines[y0 + y][x0 + x].b;
	
	scalar_t min = std::min(std::min(R,G),B);
	scalar_t max = std::max(std::max(R,G),B);
	scalar_t delta = max - min;
	
	scalar_t hue;
	if( delta == 0 ){
		return -1; 
	}
	
	if( R == max ){
		hue = ( G - B ) / delta;
	}
	else if( G == max ){
		hue = 2 + ( B - R ) / delta;
	}
	else {
		hue = 4 + ( R - G ) / delta;
	}
	hue *= 60;
	if( hue < 0 ){
		hue += 360;
	}
	
    return hue;
}
