//------------------------------------------------------------------------------
// Tasting families of features for image classification.
// 
// Copyright (c) 2011 Idiap Research Institute, http://www.idiap.ch/
// Written by Charles Dubout <charles.dubout@idiap.ch>
// 
// This file is part of Tasting.
// 
// Tasting is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.
// 
// Tasting is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with Tasting. If not, see <http://www.gnu.org/licenses/>.
//------------------------------------------------------------------------------

#include "FileInputSet.h"

#include <algorithm>
#include <cassert>
#include <fstream>
#include <iostream>
#include <iterator>

using namespace ML;
using namespace std;

FileInputSet::FileInputSet(const std::string& featuresFileName,
						   const std::string& labelsFileName,
						   const std::string& heuristicsFileName)
: featuresFileName_(featuresFileName) {
	// Try to open the files
	ifstream featuresFile(featuresFileName.c_str(), std::ios::binary);
	ifstream labelsFile(labelsFileName.c_str());
	ifstream heuristicsFile(heuristicsFileName.c_str());

	assert(featuresFile.is_open());
	assert(labelsFile.is_open());
	assert(heuristicsFile.is_open());

	// Read the labels
	copy(istream_iterator<unsigned int>(labelsFile), istream_iterator<unsigned int>(), back_inserter(labels_));

	// Read the heuristics
	copy(istream_iterator<unsigned int>(heuristicsFile), istream_iterator<unsigned int>(), back_inserter(heuristics_));

	// Find out the number of samples and features
	unsigned int nbSamples = labels_.size();

	unsigned int nbFeatures = 0;

	for (unsigned int i = 0; i < heuristics_.size(); ++i)
		nbFeatures += heuristics_[i];

	// Resize the initial stack of samples
	sampleStack_.resize(1);
	sampleStack_[0].resize(nbSamples);

	// Make the indices range from 0 to nbSamples - 1
	for(unsigned int s = 0; s < nbSamples; ++s)
		sampleStack_[0][s] = s;

	// Resize the initial stack of features
	featureStack_.resize(1);
	featureStack_[0].resize(nbFeatures);

	// Make the indices range from 0 to nbFeatures - 1
	for(unsigned int f = 0; f < nbFeatures; ++f)
		featureStack_[0][f] = f;

	// Number of labels
	unsigned int nbLabels = *max_element(labels_.begin(), labels_.end()) + 1;

	nbLabelStack_.push_back(nbLabels); // Push it twice so as to be sure to
	nbLabelStack_.push_back(nbLabels); // never overwrite it

	// Number of heuristics
	nbHeuristics_ = heuristics_.size();

	cout << "[FileInputSet::FileInputSet] #samples: " << nbSamples
		 << ", #features: " << nbFeatures << ", #labels: " << nbLabels
		 << ", #heuristics: " << nbHeuristics_ << '.' << endl;

	// Set the number of images to 1
	nbImages_ = 1;
}

void FileInputSet::matrixCache(std::vector<scalar_t>& data) const {
	// Open the features file
	ifstream featuresFile(featuresFileName_.c_str(), ios::binary);
	assert(featuresFile.is_open());

	// Get the current number of samples, features, and initial number of samples
	const unsigned int nbSamples = this->nbSamples();
	const unsigned int nbFeatures = this->nbFeatures();
	const unsigned long stride = sampleStack_[0].size() * sizeof(float);

	// Resize the vector to contain the selected subset
	data.resize(nbSamples * nbFeatures);

	// Convert the sample indices to the level 0
	std::vector<unsigned> indices(nbSamples);

	for(unsigned int s = 0; s < nbSamples; ++s) {
		register unsigned int index = s;

		for(unsigned int i = sampleStack_.size() - 1; i > 0; --i)
			index = sampleStack_[i][index];

		indices[s] = index;
	}

	// Report to the log the progress in filling the cache if there is more
	// than one sample and one feature
//	if(nbSamples > 1 && nbFeatures > 1)
//		cout << "[FileInputSet::matrixCache] Computing " << data.size() << " features...";

	unsigned int percentage = 100;

	vector<float> tmp(stride);

	// For all the selected features
	for(unsigned int f = 0; f < nbFeatures; ++f) {
		register unsigned int index = f;

		for(unsigned int i = featureStack_.size() - 1; i > 0; --i)
			index = featureStack_[i][index];

		// Load all the features of the sample
		featuresFile.seekg(index * stride, ios_base::beg);

		assert(featuresFile.read(reinterpret_cast<char*>(&tmp[0]), stride));

		for(unsigned int s = 0; s < nbSamples; ++s)
			data[s * nbFeatures + f] = tmp[indices[s]];

	//	unsigned int percent = f * 100 / nbFeatures;

	//	if(nbSamples > 1 && nbFeatures > 1 && percent != percentage)
	//		cout << ' ' << (percentage = percent) << '%' << flush;
	}

//	if(percentage != 100)
//		cout << " 100%." << endl;
}

void FileInputSet::labelCache(std::vector<unsigned int>& data) const {
	// Get the current number of samples
	const unsigned int nbSamples = this->nbSamples();

	// Resize the vector to contain the selected subset
	data.resize(nbSamples);

	// For all the selected samples
	for(unsigned int s = 0; s < nbSamples; ++s) {
		register unsigned int index = s;

		for(unsigned int i = sampleStack_.size() - 1; i > 0; --i)
			index = sampleStack_[i][index];

		data[s] = labels_[index];
	}
}

void FileInputSet::weightCache(std::vector<scalar_t>& data) const {
	// Resize the vector to contain the selected subset and fill it with 1's
	data.resize(nbSamples());
	fill(data.begin(), data.end(), 1);
}

void FileInputSet::imageCache(std::vector<unsigned int>& data) const {
	// Resize the vector to contain the selected subset and fill it with 0's
	data.resize(nbSamples());
	fill(data.begin(), data.end(), 0);
}

void FileInputSet::coordinatesCache(std::vector<coordinates_t>& data) const {
	// Resize the vector to contain the selected subset and fill it with 0's
	data.resize(nbSamples());

	coordinates_t zero;
	zero.x = 0;
	zero.y = 0;

	fill(data.begin(), data.end(), zero);
}

void FileInputSet::heuristicCache(std::vector<unsigned int>& data) const {
	// Get the current number of features
	const unsigned int nbFeatures = this->nbFeatures();

	// Resize the vector to contain the selected subset
	data.resize(nbFeatures);

	vector<unsigned int> heuristics;

	for (unsigned int i = 0; i < heuristics_.size(); ++i)
		for (unsigned int j = 0; j < heuristics_[i]; ++j)
			heuristics.push_back(i);

	// For all the selected features
	for(unsigned int f = 0; f < nbFeatures; ++f) {
		register unsigned int index = f;

		for(unsigned int i = featureStack_.size() - 1; i > 0; --i)
			index = featureStack_[i][index];

		data[f] = heuristics[index];
	}
}

