// Copyright (c) 2007 David Grangier
// Copyright (c) 2007 Samy Bengio
// 
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are 
// met: Redistributions of source code must retain the above copyright 
// notice, this list of conditions and the following disclaimer.
// Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the 
// documentation and/or other materials provided with the distribution.
// The name of the author may not be used to endorse or promote products
// derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, 
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.


#include "RetrievalEvaluation.h"

namespace Torch {

RetrievalEvaluation::RetrievalEvaluation()
{
	nrel = 0;
	rel_rank = NULL;
}

void RetrievalEvaluation::set(int ndoc_, real *scores, svector *rel_doc)
{
	ndoc = ndoc_;
	nrel = rel_doc->size;
	relDocRank(scores, rel_doc);
}

void RetrievalEvaluation::relDocRank(real *scores, svector *rel_doc)
{
	// relevant doc scores (sorted)
	sreal* rel_scores = (sreal*) allocator->alloc(nrel * sizeof(sreal));
  for (int d = 0; d < nrel; d++)
  {
		int id = rel_doc->frame[d].index;
    rel_scores[d].index = id;
    rel_scores[d].value = scores[id];
  }
  qsort(rel_scores, nrel, sizeof(sreal), sreal_comp_values_ascend);
	// initalize rel doc rank
	rel_rank = (int*) allocator->realloc(rel_rank, sizeof(int) * nrel);
	memset(rel_rank, 0, sizeof(int) * nrel);
	// determine rel doc rank
	for (int d = 0; d < ndoc; d++)
	{
		int i = 0;
		while ((i < nrel) && (scores[d] >= rel_scores[i].value))
		{
			rel_rank[nrel - 1 - i]++;
			i++;
		}
	}	
	allocator->free(rel_scores);
}

real RetrievalEvaluation::errorRate()
{
  int e = 0;
  for (int i = 0; i < nrel; i++)
    e += rel_rank[i] - 1 - i; // number of non rel doc above myself
  real res = (real) e;
  res /= (real) (nrel * (ndoc-nrel));
  return res;
}

real RetrievalEvaluation::avgp()
{
  real avgp = 0;
  for (int i = 0; i < nrel; i++)
  {
    avgp += ((real) i + 1)/((real) rel_rank[i]);
  }
  avgp /= (real) nrel;
  return avgp;
}

real RetrievalEvaluation::bep()
{
	return ptop(nrel);
}

real RetrievalEvaluation::ptop(int n)
{
	int i = 0;
	while ((i < nrel) && (rel_rank[i] <= n))
		i++; 
  return (((real) i) / ((real) n));
}

RetrievalEvaluation::~RetrievalEvaluation()
{}

}
