// Copyright (c) 2007 David Grangier
// Copyright (c) 2007 Samy Bengio
// 
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are 
// met: Redistributions of source code must retain the above copyright 
// notice, this list of conditions and the following disclaimer.
// Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the 
// documentation and/or other materials provided with the distribution.
// The name of the author may not be used to endorse or promote products
// derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, 
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.


#include "PAMachineImageIR.h"
#include "PAExample.h"

namespace Torch {

PAMachineImageIR::PAMachineImageIR(int n_term_, PAMachine **term_machine_)
{
	n_term = n_term_;
	term_machine = term_machine_;
	kernel = (n_term>0) ? term_machine[0]->kernel : NULL; 
}

void PAMachineImageIR::init()
{
	for (int t = 0; t < n_term; t++)
		term_machine[t]->init();
}

real PAMachineImageIR::forward(PAExample *x)
{
	return forward((ImgConstraint*)x);
}

real PAMachineImageIR::forward(ImgConstraint *x)
{ // Rk: negative_img can be NULL for testing
	real res = 0;

	// sum up the outputs of all term machines
	for (int t = 0; t < x->query->size; t++)
	{
		int tid = x->query->frame[t].index;
		real tval = x->query->frame[t].value;
		// forward positive and negative image in the current term machine
		real delta = term_machine[tid]->forward(x->positive_img);
		delta -= (x->negative_img) ? term_machine[tid]->forward(x->negative_img) : 0.0;
		
		res += tval * delta;
	}

	return res;
}

void PAMachineImageIR::addSupport(real alpha, PAExample *x)
{
	addSupport(alpha, (ImgConstraint*)x);
}

void PAMachineImageIR::addSupport(real alpha, ImgConstraint *x)
{ // Rk: negative_img cannot be NULL for training
	// forward the support vectors to all term machine
	for (int t = 0; t < x->query->size; t++)
	{
    int tid = x->query->frame[t].index;
    real tval = x->query->frame[t].value;

		term_machine[tid]->addSupport(alpha * tval, x->positive_img);
		term_machine[tid]->addSupport(-alpha * tval, x->negative_img);
	}
}

real PAMachineImageIR::squareNorm(PAExample *x)
{
	return squareNorm((ImgConstraint*)x);
}

real PAMachineImageIR::squareNorm(ImgConstraint *x)
{
	// query norm
	real q_norm2 = 0;
	for (int t = 0; t < x->query->size; t++)
		q_norm2 += x->query->frame[t].value * x->query->frame[t].value;
	// (p+ - p-) norm
	real p_norm2 = kernel->forward(x->positive_img, x->positive_img);
	p_norm2 += -2.0 * kernel->forward(x->positive_img, x->negative_img);
	p_norm2 += kernel->forward(x->negative_img, x->negative_img);

	return q_norm2 * p_norm2;
}

void PAMachineImageIR::loadXFile(XFile *file)
{
	for (int t = 0; t < n_term; t++)
		term_machine[t]->loadXFile(file);
}

void PAMachineImageIR::saveXFile(XFile *file)
{
  for (int t = 0; t < n_term; t++)
    term_machine[t]->saveXFile(file);
}

PAMachineImageIR::~PAMachineImageIR() {}

}
