#ifndef IP_PLANAR_TRANSFORM_HH
#define IP_PLANAR_TRANSFORM_HH


#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>
#include <iomanip.h>

#include "ip_PixelAndPoint.h" // for the definition of real

namespace ImageProcessing  {
  
  /** 
      Basic representation for 2D planar transform 
      
      @author Jean-marc Odobez (Jean-Marc.Odobez@idiap.ch)
      @author Daniel Gatica-Perez (gatica@idiap.ch)
  */
  
  /*
      Basic representation for planar transform

      pp     =  f(p,origin)
      dep(p) =  pp-p;

      p     : initial point
      pp    : image point
      dep(p): displacement at point p

    
  */

  //============================================================

  typedef enum ip_PlanarTransformModel {
    PLANAR_NULL,
    PLANAR_TRANSLATION,
    PLANAR_TRANSLATION_SCALING,
    PLANAR_TRANSLATION_SCALING2
  } ip_PlanarTransformModel ;


  class  ip_PlanarTransform
    {
    public:
      real       * m_pParams;
      ip_Point     m_cOrigin;    // defines the origin of the transform
      int          m_iStepRead;  // steps between two parameters in input
      // parameter

      // private
      int          m_iNbParams;

      //-------- 
      ip_PlanarTransform();

      //-------- 
      virtual ip_PlanarTransformModel  planarModel()=0;

      //-------- 
      virtual int    nbParams();

      //---- dgp
      virtual real *getPtrToParams(void);

      //-------- 
      virtual inline void  getParams(real * _Params,int StepWrite=1);

      //-------- 
      virtual inline  void  setParams(real * _Params){
	int i; 
	for (i=0;i<m_iNbParams;i++){
	  m_pParams[i]=*_Params;
	  _Params += m_iStepRead;
	}
      }
      
      //------------------------- 
      //- init to the null motion (identity transform)
      //- i.e. : after this step, p=pp=f(p,origin)
      virtual   void  setIdentity()=0;

      //------------------
      // Add a translation
      //  Before translate : pp    = fb(p,origin)
      //  after  translate : newpp = fa(p,origin) = fb(p,origin)+dep
      // dep=(depx,depy)
      virtual   void  translate(real depx,real depy);


      //-------- 
      // 
      virtual   void  setOrigin(ip_Point new_origin);

      //-------- 
      // When changing the origin, the transform should
      // not change (i.e. image of pp remains the same
      //  before and after applying changeOrigin())
      virtual   void  changeOrigin(ip_Point new_origin)=0;

      //--------
      // x = column number, y line number
      //  image(p) = p + dep(p)
      inline virtual ip_Point imageReal(double x, double y) =0; 


      //--------
      inline virtual ip_Point   imagePix(ip_Pixel p){
	return imageReal((real)p.m_iCo,(real)p.m_iLi);}

      //--------
      inline virtual ip_Point   imagePoint(ip_Point p){ 
	return  imageReal(p.m_rX,p.m_rY); }

      //--------
      //  C = column number, L line number
      inline virtual ip_Point   imageDiscrete(int  C,int L){
	return imageReal((real)C,(real)L); }


      //-------   dep =  displacement
      //  x = column number, y line number
      inline virtual ip_Vector depReal(real x, real y){
	ip_Point p=imageReal(x,y);
	p.m_rX-=x; p.m_rY-=y;
	return p;
      }

      //--------
      inline virtual ip_Vector depPix(ip_Pixel p){ 
	return depReal((real)p.m_iCo,(real)p.m_iLi);}

      //--------
      inline virtual ip_Vector depPoint(ip_Point p){ 
	return  depReal(p.m_rX,p.m_rY); }

      //--------
      //  C = column number, L line number
      inline virtual ip_Vector depDiscrete(int  C,int  L){
	return depReal((real)C,(real)L); }
       
      //-------
      virtual void display(char * msg) =0;
  
      //-------
      virtual ~ip_PlanarTransform();
      
};

}

#endif  // ip_PLANAR_TRANSFORM_HH
