#ifndef IP_MULTI_IMAGE_INC
#define IP_MULTI_IMAGE_INC

//#include "general.h"

#include "ip_Image.h"
#include "ip_ColorDef.h"

namespace ImageProcessing {

  //#define DEBUG_CLASSES 0

  //-----
  
  /** 
   
      @author Jean-marc Odobez (Jean-Marc.Odobez@idiap.ch)
      @author Daniel Gatica-Perez (gatica@idiap.ch)
  */
  //===================================================
  //
  //    class         ip_BaseMultiImage
  //
  //    Base virtual class for representation
  //    of set of images with different sizes 
  //
  //    Each image is called a "band"
  //
  //    1/05/2002:
  //    
  //
  //===================================================

  //===================================================
  template <class Type>
    class  ip_BaseMultiImage {
    public:
    //----------------
    // informations members
    //
    // band index : from 0 to nbBands-1
    // line index : from 0 to nbLines-1
    // col  index : from 0 to nbColumns-1
    virtual inline int nbBands()=0;
    virtual inline int nbLines(int k)=0;    // number of lines in band k
    virtual inline int nbColumns(int k)=0; // number of columns in band k
   
    //----------------
    // managing memory and initialisation

    virtual void allocateMem(int nb_bands, int *nbli,int *nbco)=0; 
    // (re) allocate memory, if necessary
    virtual void allocateMemBand(int k, int nbli,int nbco)=0; 
    // (re) allocate memory, if necessary
    
    virtual void freeMem(int k)=0; // destroy memory AND corresponding 
    // structure from band k to nbBands()-1
    // may need to do some reallocation
    virtual void freeMemBand(int k)=0; // destroy memory in band k

    //------------------
    virtual void init(Type value){    // initialisation to value in all bands
      int i;
      for(i=0;i<nbBands();i++) init(i,value);
    }
    //------------------
    virtual void init(Type *value){    // initialisation to value 
      int i;
      for(i=0;i<nbBands();i++) init(i,value[i]);
    }

    //------------------
    virtual void init(int k,Type value)=0;  //  initialisation in band k

    // virtual void permute(ip_Image<Type> & tab); // permutation of images (without
    // recopy of content


    //------------------
    // accessing elements
    // Note that normally, the () operator and value fonction are faster
    // than the setVal or getVal functions

    virtual inline Type & operator()(int k,int li,int co) = 0; // M(i,j)=a; or a=M(i,j);
    virtual inline Type & value(int k,int li,int co) = 0;  // M.value(i,j)=a; or a=M.value(i,j);
    virtual inline Type & operator()(int k,int li,int co) const = 0;
    virtual inline Type & value(int k,int li,int co) const = 0; 

    virtual inline void setVal(int k,int li,int co,Type val) =0;
    virtual inline Type getVal(int k,int li,int co) =0;

    // accessing elements after having set the default band (cf setBand())
    virtual inline Type & operator()(int li,int co) = 0; // M(i,j)=a; or a=M(i,j);
    virtual inline Type & value(int li,int co) = 0;    // M.value(i,j)=a; or a=M.value(i,j);
    virtual inline Type & operator()(int li,int co) const = 0;
    virtual inline Type & value(int li,int co) const = 0; // M.value(i,j)=a; or a=M.value(i,j);
    virtual inline void setVal(int li,int co, Type val) =0;
    virtual inline Type getVal(int li,int co) =0;
    
    //------------------
    // sweeping through image
    // Setting image line, and then accessing through
    // column number

    // set the default row in all bands
    // (and NOT in the default band)
    virtual inline void setLine(int li){
      int i;
      for(i=0;i<nbBands();i++) setBandLine(i,li); 
    }
    virtual inline void setBandLine(int k,int li) = 0;

    virtual inline Type & valueBandCol(int k,int co) = 0; 
    virtual inline Type & valueBandCol(int k,int co) const = 0; 

    // to read individuals from  the default band, default line
    virtual inline void setBand(int k)=0;
    virtual inline Type & valueCol(int co) = 0; 
    virtual inline Type & valueCol(int co) const = 0; 


    virtual ~ip_BaseMultiImage(){}
    
  };
  
  //====================================================
  //
  //    class ip_MultiImage    
  //
  // An implementation that should work with 
  //  any implementation of ip_Image
  //
  //
  // Added member function : 
  //
  //      - convert(ip_Image<ip_ColorElement8u> & Ima)
  //        converts a color image into a 3 bands image
  //        (WARNING : memory (re)allocation is performed
  //
  //====================================================
  template<class Type>
    class  ip_MultiImage :
    public  ip_BaseMultiImage<Type> {
    private:
      ip_Image<Type>          **m_pImagePtr;
      int                       m_iNbBands;
      ip_Image<Type>           *m_pCurrentBand;
      
    public:
      //-----------------------------
      // constructors : must not call
      //  allocateMem
      //-----------------------------
      ip_MultiImage(){
	m_iNbBands=0; m_pCurrentBand=NULL; m_pImagePtr=NULL;
	}

      //----------------
      // informations members
      //
      // band index : from 0 to nbBands-1
      // line index : from 0 to nbLines-1
      // col  index : from 0 to nbColumns-1
      virtual inline int nbBands(){return m_iNbBands;}
      virtual inline int nbLines(int k){ return m_pImagePtr[k]->nbLines();}
      virtual inline int nbColumns(int k){ return m_pImagePtr[k]->nbColumns();}

      //--- Number of lines of the current band
      virtual inline int nbLines(){ 
	if(m_pCurrentBand!=NULL)
	  return m_pCurrentBand->nbLines();
	else return 0;
      }
      virtual inline int nbColumns(){ 
	if(m_pCurrentBand!=NULL)
	  return m_pCurrentBand->nbColumns();
	else
	  return 0;
      }
      
      //----------------
      // managing memory and initialisation
      //
      //
      // require to define the allocation of a new ip_Image
      //----------------
      virtual ip_Image<Type> * newIpImage()=0;
      

      //----------------
      //
      // warning : allocateMem can not be called before this object has
      // been created
      // 
      //

      //----------------
      virtual void allocateMem(int nbbands, int nbli,int nbco){
	int *nbl=NULL,*nbc=NULL,i;
	if(nbbands>0) {
	  nbl = new int [nbbands];
	  nbc = new int [nbbands];
	}
	for(i=0;i<nbbands;i++){ nbl[i]=nbli; nbc[i]=nbco;}
	allocateMem(nbbands,nbl,nbc);
	if(nbbands>0) { delete [] nbl; delete [] nbc; }
      }
      
      //----------------
      virtual void allocateMem(int nb_bands, int *nbli,int *nbco){
	// OK : easyest thing would be to deallocate everything and reallocate
	//
	int k;

#ifdef  DEBUG_CLASSES
	printf("ipMultiImage : allocateMem(int, *int , *int) : 1 \n"); fflush(stdout);
#endif

	if(nb_bands!=nbBands()){
	  ip_Image<Type>    **temp=NULL;
	  int               m=nbBands();
	  if(nb_bands<m) m=nb_bands;
	  
	  if(nb_bands>0)
	    temp = new ip_Image<Type> * [nb_bands];
	  // recopy of pointers
	  for(k=0;k<m;k++)
	    temp[k]=m_pImagePtr[k];
	  // if less bands are asked for, destroy bands
	  for(k=m;k<nbBands();k++)
	    delete m_pImagePtr[k];
	  // if more bands are needed, create new ones
	  for(k=m;k<nb_bands;k++)
	    temp[k] = newIpImage();

	  // destroy
	  delete [] m_pImagePtr;
	  m_pImagePtr=temp;
	}
	// to finish, provide new dimension
	m_iNbBands=nb_bands;
	for(k=0;k<m_iNbBands;k++)
	  allocateMemBand(k,nbli[k],nbco[k]);

	// check : if current band is NULL -> set to first band
	if(m_pCurrentBand==NULL)
	  setBand(0);

      } //====== end allocateMem


      //----------------
      virtual void allocateMemBand(int k, int nbli,int nbco){

#ifdef  DEBUG_CLASSES
	printf("ipMultiImage : allocateMemBand(k=%d,nbli=%d,nbco=%d) : 1 \n",k,nbli,nbco); fflush(stdout);
#endif

	if(k>=m_iNbBands){
	  printf("\n\nWarning\nWarning : trying to allocate memory in non existing band\n");
	  fflush(stdout);
	}
	else 
	  m_pImagePtr[k]->allocateMem(nbli,nbco);
      }
      
      //----------------
      virtual void freeMem(int from_band){ // destroy memory AND corresponding 
	// structure from band from_band to nbBands()-1
	// may need to do some reallocation
	int k;
	
	if(nbBands()>0){
	  ip_Image<Type>    **temp=NULL;
	  if(from_band<nbBands() && from_band!=0){
	    temp = new ip_Image<Type> * [from_band];
	    // recopy of pointers to images
	    for(k=0;k<from_band;k++)  temp[k]=m_pImagePtr[k];
	  }
	  for(k=from_band;k<nbBands();k++)
	    delete m_pImagePtr[k];
	  delete [] m_pImagePtr;
	  m_pImagePtr=temp;
	  m_iNbBands=from_band;
	}
      }
      
      //----------------
      virtual void freeMemBand(int k){ // destroy memory in band k
	m_pImagePtr[k]->freeMem();
      }
      
      //------------------
      //
      // Initialization
      //
      //------------------
      virtual void init(int k,Type value){ //  initialisation in band k
	if(k>=nbBands()){
	  printf("\n\nWarning\nWarning : initialisation in non existing band\n");
	  fflush(stdout);
      }
	else 
	  m_pImagePtr[k]->init(value);
      }
      
      // virtual void permute(ip_Image<Type> & tab); // permutation of images (without
      // recopy of content


      //------------------
      // accessing elements
      // Note that normally, the () operator and value fonction are faster
      // than the setVal or getVal functions

      virtual inline Type & operator()(int k,int li,int co){
	return m_pImagePtr[k]->value(li,co);}
      virtual inline Type & value(int k,int li,int co){
	return m_pImagePtr[k]->value(li,co);}
      virtual inline Type & operator()(int k,int li,int co) const{
	return m_pImagePtr[k]->value(li,co);}
      virtual inline Type & value(int k,int li,int co) const {
	return m_pImagePtr[k]->value(li,co);}
      
      virtual inline void setVal(int k,int li,int co,Type val) {
	m_pImagePtr[k]->setVal(li,co,val);    }
      virtual inline Type getVal(int k,int li,int co){
	return m_pImagePtr[k]->getVal(li,co); }
      
      // accessing elements after having set the default band (cf setBand())
      virtual inline Type & operator()(int li,int co){
	return m_pCurrentBand->value(li,co); }
      virtual inline Type & value(int li,int co){
	return m_pCurrentBand->value(li,co); }
      virtual inline Type & operator()(int li,int co) const {
	return m_pCurrentBand->value(li,co); } 
      virtual inline Type & value(int li,int co) const {
	return m_pCurrentBand->value(li,co); }
      
      virtual inline void setVal(int li,int co, Type val){
	m_pCurrentBand->setVal(li,co,val); }
      virtual inline Type getVal(int li,int co){
	return m_pCurrentBand->getVal(li,co);
      }
    
      //-----------------------------------
      // Accessing the individual ip_Image
      virtual ip_Image<Type> * getIpImage(int k){
	return m_pImagePtr[k];
      }

	//------------------
	// sweeping through image
	// Setting image line, and then accessing through
	// column number
	
	virtual inline void setBandLine(int k,int li){
	m_pImagePtr[k]->setLine(li); }
      
      virtual inline Type & valueBandCol(int k,int co){
	return m_pImagePtr[k]->valueCol(co); }
      virtual inline Type & valueBandCol(int k,int co) const  {
	return m_pImagePtr[k]->valueCol(co); }
      
      // to read individuals from  the default band, default line
      virtual inline void setBand(int k){
	m_pCurrentBand=m_pImagePtr[k]; }
      virtual inline Type & valueCol(int co){
	return m_pCurrentBand->valueCol(co); }
      virtual inline Type & valueCol(int co) const {
	return m_pCurrentBand->valueCol(co); }


      //----------------------------------
      // converts from interleaved color elements 
      // to 3 bands
      // if _AllocateMem = true, reallocation is performed
      // to match input size and 3 bands.
      virtual void  convert(ip_Image<ip_ColorElement8u> & Ima,bool _AllocateMem=false){
	int li,co;

	if(_AllocateMem)
	  allocateMem(3,Ima.nbLines(),Ima.nbColumns());

	// recopy in the different bands
	for(li=0;li<Ima.nbLines();li++){
	  Ima.setLine(li);
	  setLine(li);
	  for(co=0;co<Ima.nbColumns();co++){
	    valueBandCol(0,co)=(Type)Ima.valueCol(co).one;
	    valueBandCol(1,co)=(Type)Ima.valueCol(co).two;
	    valueBandCol(2,co)=(Type)Ima.valueCol(co).three;
	  }
	}
      }
     
      //----------------------------------
      virtual ~ip_MultiImage(){freeMem(0);}
      
  };  
}

#endif
