// Copyright (c) 2015 Idiap Research Institute, http://www.idiap.ch/
// Written by Fabio Valente <fabio.valente@idiap.ch>
// Written by Deepu Vijayasenan <dvijayasenan@lsv.uni-saarland.de>
// Written by David Imseng <david.imseng@idiap.ch>
// Written by Srikanth Madikeri <srikanth.madikeri@idiap.ch>
//
// This file is part of the IB Speaker Diarization Toolkit.
//
// IB diarization toolkit is free software: you can redistribute it
// and/or modify it under the terms of the GNU General Public License
// version 3 as published by the Free Software Foundation.
//
// The IB Speaker Diarization Toolkit is distributed in the hope that it
// will be useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with the IB Speaker Diarization Toolkit. If not, see
// <http://www.gnu.org/licenses/>.


#include <fstream>
#include <sstream>
#include <iostream>
#include <string>
#include <vector>
#include <map>

using namespace std;


#include "featconfig.h"
#include "global.h"
#include "rhmm.h"
#define LINE_LENGTH 1024

using namespace gaussrealignment;
using namespace hmmrealignment;

int dorealign (ConfigVars& configVars)
{
   // Calling convention
   // realign --config <config File> --id <meeting id>


   int* seg_ptr; //List of begin/end times of speech segments
   int num_segs; //Num of segments 

   float *gmm_wts = NULL;

   int *clusterIDs   = NULL; //1D array mapping index => cluster ids;
   int *clusterIdx   = NULL; //1D array mapping cluster ids => indices;
   int *cluster_freq = NULL; //number of frames per cluster

   stringstream ss(stringstream::in|stringstream::out); //string object to keep the file names
   int i, j;    

   //-------------- READ THE SCP FILE ---------------
   ss << configVars.m_tmpDir << "/"<<configVars.id<<".scp";
   FILE *segfp = fopen(ss.str().c_str(),"r");
   if(segfp == NULL )
   {
      cerr<<"Unable to open scp file (generated by aibfeat) \""<<ss.str().c_str()<<"\""<<endl;
      exit(0);
   }
   fclose(segfp);

   // First create segments of maximum max_dur long 
   //Required Parameters configVars.m_maxDur
   // -- read the scp file
   realign_read_scp_file(ss.str().c_str(), &seg_ptr, &num_segs);
   //int last_speech_frame = seg_ptr[2*num_segs-1];
   ss.str(""); //Flush that buffer  for other things

   //-------------- READ THE AIB OUTPUT ---------------
   ss << configVars.m_tmpDir << "/"<<configVars.id<<".clust.out";
   FILE *clstfp = fopen(ss.str().c_str(),"r");
   if(clstfp == NULL )
   {
      cerr<<"Unable to open cluster output file (generated by IB clustering) \""<<ss.str().c_str()<<"\""<<endl;
      exit(0);
   }
   ss.str(""); //Flush that buffer  for other things

   int clstid = -1 ; //Speaker cluster id -read from the file

   ++(configVars.m_maxClusters);
   vector<int> seg_to_clustmap; 
   vector<bool> is_cluster_valid(configVars.m_maxClusters,false);
   //Finding the clusters.
   i =0;
   while(fscanf(clstfp,"%d",&clstid) != EOF)
   {
      seg_to_clustmap.push_back(clstid);
      if (clstid > configVars.m_maxClusters)
      {
         fprintf(stderr, "Error: Cluster value > max_cluster");
         exit(-1);
      }
      is_cluster_valid[clstid] = true;
      ++i;
      if(i>=num_segs)  
         break;
   }
   fclose(clstfp);
   //Finding the number of clusters
   int num_clsters = 0;
   for (i=0; i<configVars.m_maxClusters; ++i )
   {
      if(is_cluster_valid[i])
         ++num_clsters;
   }
   
   //need a mapping between cluster IDs and 
   // cluster indices

   clusterIDs = new int [num_clsters+1];
   clusterIdx = new int [configVars.m_maxClusters+1];

   j = 0;
   for (i=0; i<=configVars.m_maxClusters; ++i )
   {
      if(is_cluster_valid[i])
      {
         clusterIdx[i] = j;
         clusterIDs[j++] = i;
      }
      else
         clusterIdx[i] = -1;
   }


   //-------------- READ THE POSTERIOR FILE  ---------------
   // Read the posterior file and compute each cluster posteriors
   // This computed posteriors are used to initialize the KL HMM


   //-------- Checking the posterior input file -------

   int tot_num_frames = 0;
   int featdim = 0;

   ss << configVars.m_tmpDir << "/"<<configVars.id<<".post.all";
   FeatFileReader fileReader(ss.str().c_str(), seg_ptr, num_segs, featdim, tot_num_frames);
   ss.str(""); //Flush that buffer  for other things


   //-------- Initialize the HMM segmentation ---------
   chunk *first = NULL;
   chunk *curr = NULL;
   int start_curr_seg = 0;
   first = new chunk;
   curr = first;

   for (i=0; i<num_segs; ++i)
   {
      curr->from = start_curr_seg;
      curr->to= curr->from +(seg_ptr[2*i+1]-seg_ptr[2*i]);
      curr->cluster = clusterIdx[seg_to_clustmap[i]];
      start_curr_seg = curr->to + 1;
      if(i == (num_segs-1))
         curr->next = NULL;
      else{
         curr->next = new chunk;
         curr = curr->next;
      }
   }

   HMM hmm(num_clsters, 1, featdim, configVars.m_minDur);
   fprintf(stderr, "Initialize HMM\n");

   for (i= 0; i <NLOOPS_ADAPT; ++i)
   {
      printf("training the rkl hmm \n");
      hmm.Train_HMM(fileReader, first);   // Retrain the HMMs
      if (first != NULL)
         delete first;
      first = NULL;
      hmm.segment(fileReader, &first);  
   }

   // Need to write the segmentation out as rttm 

   ss << configVars.m_tmpDir << "/"<<configVars.id<<".rttm";
   hmm.WriteRttm((ss.str()).c_str(), first, fileReader, tot_num_frames, configVars.id.c_str());

   // delete the memory
   delete first; first = NULL;
   delete [] clusterIdx;
   delete [] clusterIDs;
   delete [] cluster_freq;
   delete [] gmm_wts;
   free(seg_ptr);
   return 0;
}

