// Copyright (c) 2015 Idiap Research Institute, http://www.idiap.ch/
// Written by Fabio Valente <fabio.valente@idiap.ch>
// Written by Deepu Vijayasenan <dvijayasenan@lsv.uni-saarland.de>
// Written by David Imseng <david.imseng@idiap.ch>
// Written by Srikanth Madikeri <srikanth.madikeri@idiap.ch>
//
// This file is part of the IB Speaker Diarization Toolkit.
//
// IB diarization toolkit is free software: you can redistribute it
// and/or modify it under the terms of the GNU General Public License
// version 3 as published by the Free Software Foundation.
//
// The IB Speaker Diarization Toolkit is distributed in the hope that it
// will be useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with the IB Speaker Diarization Toolkit. If not, see
// <http://www.gnu.org/licenses/>.


#include "featfileread.h"
#include <iostream>

using namespace std;
// Default constructor
// Just initializing everything to NULL
// In case it gets called later this should tell
// nothing is initialized
FeatFileReader::FeatFileReader()
{
    m_featfp = NULL;
    m_indexMap = NULL;
    m_byte_per_frame = 0;
    m_featdim = 0;
    m_headersize = 0;
    m_nSamples =0;
}

/* This function opens the posterior file,
 * read the feature file, read the segment info
 * and create the map m_indexMap[i] should 
 * give i^th speech frame 
 */
FeatFileReader::FeatFileReader(const char*featfilename, 
                               const int* seg_info, int num_segs, 
                               int& featdim, int& num_fr)
{
    //HTK Header 
    unsigned int n_frames,s_rate;
    __uint16_t tc; 

    m_indexMap = new int[seg_info[2*num_segs -1]+1]; //That is just an upper bound

    // DEBUG DEEPU
    printf("last speech frame = %d ", seg_info[2*num_segs -1]);

    for(int i=0; i < seg_info[2*num_segs -1]+1;++i)
        m_indexMap[i] = -1;

    m_featfp = NULL;
    m_featfp = fopen(featfilename,"rb");
    if(m_featfp ==NULL ){
        printf("Cannot open posterior file %s (Generated by aibfeat)",featfilename);
    }

    //Reading Header
    fread(&n_frames,SIZE_LONG,1,m_featfp); //Number of frames
    fread(&s_rate,SIZE_LONG,1,m_featfp); //Sampling Rate
    fread(&m_byte_per_frame,SIZE_SHORT,1,m_featfp);//#byte_per_frame
    fread(&tc,SIZE_SHORT,1,m_featfp); //tc ??
    m_headersize = 2*(SIZE_LONG+SIZE_SHORT);

    m_byte_per_frame = SwapByteOrderOfShort(&m_byte_per_frame);
    tc = SwapByteOrderOfShort(&tc);
    n_frames = SwapByteOrderOfLong(&n_frames);
    s_rate = SwapByteOrderOfLong(&s_rate);

    m_featdim = featdim = m_byte_per_frame / SIZE_LONG; //assigning output
    m_sampleRate = s_rate;

    num_fr = -1;
    int seg_index = 0;
    unsigned int first_index =seg_info[0];
    unsigned int last_index =seg_info[1];
    for (unsigned int frame_index = 0; frame_index < n_frames; ++frame_index)
    {

        //If the frame is within the segment
        if((frame_index>=first_index) &&(frame_index<=last_index))
        {
            m_indexMap[++num_fr] = frame_index;
            //std::cout << "frame: " << num_fr << " index: : " << frame_index << std::endl;
            if(frame_index == last_index)
            {
                //Moving to a new segment index 
                //Get the mixture index of the gaussian
                ++seg_index ;  //The segment index
                if(seg_index >= num_segs) break; 
                first_index = seg_info[seg_index*2];  //Start time of seg
                last_index  = seg_info[seg_index*2 +1];//End time of seg

            }
        }
    }

    m_nSamples = ++num_fr ; //since it started from -1
    num_fr = n_frames; //returning the total number of samples :)
    printf("total number of speech frames = %d\n",m_nSamples);
}

int FeatFileReader::get_nSamples()
{
   return m_nSamples;
}

int FeatFileReader::getSampleRate()
{
   return m_sampleRate;
}

//This function accepts a speech only index
// and map it to the actual index
int FeatFileReader::findFrameIndex(int spIdx)
{
   return (spIdx < m_nSamples )?m_indexMap[spIdx]:-1;
}

//Destructor, close file, free the array
FeatFileReader::~FeatFileReader()
{
   if (m_featfp != NULL )
      fclose (m_featfp);
   if (m_indexMap != NULL )
      delete [] m_indexMap;
}

//This function read the feature file,
//Unfortunately it is always a fseek 
// followed by fread and swapping the frames

void FeatFileReader::readfeat(int fr_index, float* featvec)
{
   //std::cout << fr_index << std::endl;
   float *frame_ptr = NULL;
   long int offset = m_headersize + ( (m_indexMap[fr_index]) * m_byte_per_frame); 
   if ( fseek(m_featfp, offset,SEEK_SET) != 0) {
	  std::cout << "ERROR!!!!" << std::endl;
   }

    //Reading entire frame
   fread(featvec,SIZE_LONG,m_featdim,m_featfp);


   //Byte swapping all the frames
   for (frame_ptr = featvec ; frame_ptr != featvec+m_featdim; ++frame_ptr)
   	*frame_ptr = SwapByteOrderOfFloat(frame_ptr);
}
