"""
A wrapper around the Viola-Jones face detector from the OpenCV
library.

Copyright (c) 2011 Carl Scheffler <carl.scheffler@gmail.com>

This file is distributed as part of FaceColorModel.

FaceColorModel is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License version 3 as
published by the Free Software Foundation.

FaceColorModel is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with FaceColorModel. If not, see <http://www.gnu.org/licenses/>.
"""
import opencv

# Extract path of this module
my_path = __file__
pos = __file__.rfind('/')
if pos != -1:
    my_path = my_path[:pos+1]
else:
    my_path = ''

# Global Variables
cascade = None
storage = opencv.cvCreateMemStorage(0)
cascade_name = my_path + 'haarcascade_frontalface_alt.xml'

# Parameters for haar detection
# From the API:
# The default parameters (scale_factor=1.1, min_neighbors=3, flags=0) are tuned 
# for accurate yet slow object detection. For a faster operation on real video 
# images the settings are: 
# scale_factor=1.2, min_neighbors=2, flags=CV_HAAR_DO_CANNY_PRUNING, 
# min_size=<minimum possible face size

parameters = {
    'robust': { # Settings for slow but robust detection--tuned for filtering web images
        'min_size': opencv.cvSize(20,20),
        'image_scale': 1.1,
        'window_scale': 1.1,
        'min_neighbors': 3,
        'haar_flags': 0,
    },
    'webcam': { # Settings for faster detection--tuned for webcam input
        'min_size': opencv.cvSize(80,80),
        'image_scale': 1.2,
        'window_scale': 1.1,
        'min_neighbors': 2,
        'haar_flags': 0,
    },
}

def viola_jones_opencv(iImage, method='robust', iParam=None):
    """
    Find the largest face in an input image using the Viola-Jones face
    detector.
    
    Inputs:
      iImage is A CvMat or IplImage object that will be passed to the
        face detector.
      iParam is A dictionary of parameters that are passed as
        arguments to the face detector. The input image is downscaled
        by the factor 'image_scale' before being passed to the face
        detector. See the OpenCV documentation on the
        cvHaarDetectObjects() function for the meaning of the
        remaining dictionary entries.

    Returns: A bounding box in matrix coordinates, (first row (y),
      first column (x), last row (y), last col (x)).
    """
    global cascade, parameters
    if iParam is None:
        iParam = parameters[method]
        
    small_size = [int(round(iImage.width/iParam['image_scale'])), int(round(iImage.height/iParam['image_scale']))]
    small_img = opencv.cvCreateImage(opencv.cvSize(small_size[0], small_size[1]), 8, 1)
    opencv.cvResize(iImage, small_img, opencv.CV_INTER_LINEAR)
    opencv.cvClearMemStorage(storage)
    faces = opencv.cvHaarDetectObjects(small_img, cascade, storage, iParam['window_scale'],
                                       iParam['min_neighbors'], iParam['haar_flags'], iParam['min_size'])
    if faces and (faces.total > 0):
        # find largest face
        max_size = -1
        for i in range(faces.total):
            if faces[i].width > max_size:
                best_face_index = i
                max_size = faces[i].width
        r = faces[best_face_index]
        true_scale = [iImage.width / float(small_size[0]), iImage.height / float(small_size[1])]
        return (r.y * true_scale[1], r.x * true_scale[0], (r.y+r.height) * true_scale[1], (r.x+r.width) * true_scale[0])
    return None

cascade = opencv.cvLoadHaarClassifierCascade(cascade_name, opencv.cvSize(1,1))
if not cascade:
    print "ERROR: Could not load classifier cascade"
    import sys
    sys.exit(-1)
