// Copyright (c) 2007 David Grangier
// Copyright (c) 2007 Samy Bengio
// 
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are 
// met: Redistributions of source code must retain the above copyright 
// notice, this list of conditions and the following disclaimer.
// Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the 
// documentation and/or other materials provided with the distribution.
// The name of the author may not be used to endorse or promote products
// derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, 
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.


#ifndef S_KERNEL_INC
#define S_KERNEL_INC

#include "Object.h"
#include "Kernel.h"
#include "DiagonalGMM.h"
#include "int_real.h"

namespace Torch {

class FeatureNormKernel : public Kernel
{
  public:
    ///
		Kernel* kernel;
    FeatureNormKernel(Kernel* kernel_);

    /// Compute kernel between the example #x# and #y#.
    virtual real eval(Sequence *x, Sequence *y);
    //-----

    virtual ~FeatureNormKernel();
};


class GMMKernel : public Kernel
{
  public:
    ///
		Kernel* kernel;
		DiagonalGMM* cluster;

    GMMKernel(Kernel* kernel_, DiagonalGMM* cluster_);


    /// Compute kernel between the example #x# and #y#.
    virtual real eval(Sequence *x, Sequence *y);
    //-----

    virtual ~GMMKernel();
};

class MaxKernel : public Kernel
{
  public:
    ///
		bool symetric;
		Kernel* kernel;

    MaxKernel(Kernel* kernel_, bool symetric_=true);

		void setSymetric(bool symetric_);

    /// Compute kernel between the example #x# and #y#.
    virtual real eval(Sequence *x, Sequence *y);
    //-----

    virtual ~MaxKernel();
};


class DDKernel : public Kernel
{
  public:
		Sequence* x_cache;
		Sequence* y_cache;
		int win_size;
		bool symetric;
    ///
		Kernel* kernel;
    DDKernel(Kernel* kernel_, int line_width_, int line_height_, int win_size_=2, bool symetric_=true);


    /// Compute kernel between the example #x# and #y#.
    virtual real eval(Sequence *x, Sequence *y);
		virtual void compute_cache(Sequence *x, Sequence *y, Sequence* cache);
		virtual void saveXFile(XFile* xf);
    //-----

    virtual ~DDKernel();
};

class HausdorffKernel : public Kernel
{
  public:
    ///
		Kernel* kernel;
    HausdorffKernel(Kernel* kernel_,real n_best_=1.);
		real n_best;


    /// Compute kernel between the example #x# and #y#.
    virtual real eval(Sequence *x, Sequence *y);
    //-----

    virtual ~HausdorffKernel();
};


/** Gaussian Kernel $k(x,y) = exp(-r*g * ||x-y||^2)$ with r=1 if y==-1 and r=r if y==1
    
    @author Johnny Marithoz (marietho@idiap.ch)
*/
class RatioGaussianKernel : public Kernel
{
  public:
    real g;
		real r;
		DataSet* data;

    ///
    RatioGaussianKernel(real g_, real ratio=1.);

		/// Suppose that x is the inputs and y the support vectors
		virtual void setDataSet(DataSet* data_);
    virtual real eval(Sequence *x, Sequence *y);
    virtual ~RatioGaussianKernel();
};


class MeanKernel : public Kernel
{
  public:
    ///
		Kernel* kernel;
		Sequence* current_sequence;

    MeanKernel(Kernel* kernel_);


    /// Compute kernel between the example #x# and #y#.
    virtual real eval(Sequence *x, Sequence *y);

    virtual ~MeanKernel();
};

class WinMaxKernel : public MeanKernel
{
  public:
    ///
		int win_size;

    WinMaxKernel(Kernel* kernel_, int win_size);


    /// Compute kernel between the example #x# and #y#.
    virtual real eval(Sequence *x, Sequence *y);
    //-----

    virtual ~WinMaxKernel();
};
class ClusterKernel : public MeanKernel
{
  public:
    ///
		int win_size;

    ClusterKernel(Kernel* kernel_, int win_size_ = 1);


    /// Compute kernel between the example #x# and #y#.
    virtual real eval(Sequence *x, Sequence *y);
    //-----

    virtual ~ClusterKernel();
};

class GammaKernel : public MeanKernel
{
  public:
    ///
		int n_best;
		int n_gamma;
		Int_real** x_sort;
		Int_real** y_sort;

    GammaKernel(Kernel* kernel_, int n_gamma, int n_best);


    /// Compute kernel between the example #x# and #y#.
    virtual real eval(Sequence *x, Sequence *y);
    virtual void sortByGamma(Sequence *x, Sequence *y);
    //-----

    virtual ~GammaKernel();
};

}
#endif
