// Copyright (c) 2007 David Grangier
// Copyright (c) 2007 Samy Bengio
// 
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are 
// met: Redistributions of source code must retain the above copyright 
// notice, this list of conditions and the following disclaimer.
// Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the 
// documentation and/or other materials provided with the distribution.
// The name of the author may not be used to endorse or promote products
// derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, 
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.


#include "SequenceKernel.h"
//#include "gsl_sf_exp.h"
//#include <ipps.h>
#include "IOAscii.h"

namespace Torch {

//===================================================
//
//===================================================

FeatureNormKernel::FeatureNormKernel(Kernel* kernel_)
{
	kernel = kernel_;
}
real FeatureNormKernel::eval(Sequence *x, Sequence *y)
{
	real xx = kernel->eval(x,x);
	real yy = kernel->eval(y,y);
	real xy = kernel->eval(x,y);
	return xy/sqrt(xx*yy);
}

FeatureNormKernel::~FeatureNormKernel()
{
}


//===================================================
//
//===================================================

MeanKernel::MeanKernel(Kernel* kernel_)
{
	kernel = kernel_;
	current_sequence = NULL;
}



real MeanKernel::eval(Sequence *x, Sequence *y)
{
	Sequence new_x(1,x->frame_size);
	Sequence new_y(1,y->frame_size);
	real* temp_x = new_x.frames[0];
	real* temp_y = new_y.frames[0];
	if(current_sequence != NULL)
		allocator->free(current_sequence);

	current_sequence = new(allocator)Sequence(x->n_frames,y->n_frames);

	real tmp_sum = .0;
	real sum = .0;

	for(int i=0;i<x->n_frames;i++){
		tmp_sum = .0;
		new_x.frames[0] = x->frames[i];
		for(int j=0;j<y->n_frames;j++){
			new_y.frames[0] = y->frames[j];
			current_sequence->frames[i][j] = kernel->eval(&new_x, &new_y);
			//message("eval: %g",current_sequence->frames[i][j]);
			tmp_sum += current_sequence->frames[i][j];
		}
		sum += tmp_sum/(real)y->n_frames;
	}
	
	new_x.frames[0]= temp_x;
	new_y.frames[0] = temp_y;
	return sum/(real)(x->n_frames);
}

MeanKernel::~MeanKernel()
{
}

//===================================================
//
//===================================================

MaxKernel::MaxKernel(Kernel* kernel_, bool symetric_)
{
	kernel = kernel_;
	setSymetric(symetric_);
}

void MaxKernel::setSymetric(bool symetric_)
{
	symetric = symetric_;
}
real MaxKernel::eval(Sequence *x, Sequence *y)
{
	Sequence new_x(1,x->frame_size);
	Sequence new_y(1,y->frame_size);
	real* temp_x = new_x.frames[0];
	real* temp_y = new_y.frames[0];

	real sum_x = .0;

	
	//allocate memory for max value for y
	real* max_y = (real*)Allocator::sysAlloc(sizeof(real)*y->n_frames);
	for(int j=0; j<y->n_frames;j++)
			max_y[j] = -INF;

	for(int i=0;i<x->n_frames;i++){

		new_x.frames[0] = x->frames[i];
		real max_x = -INF;

		for(int j=0;j<y->n_frames;j++){
			
			new_y.frames[0] = y->frames[j];
			
			real val = kernel->eval(&new_x, &new_y);
			if (val> max_x){
				max_x = val;
			}
			if(val > max_y[j])
				max_y[j] = val;
		
		}
		
		sum_x += max_x;
	}
	
	real sum_y = .0;
	for(int j=0; j<y->n_frames;j++)
			sum_y += max_y[j];
	
	free(max_y);
	
	new_x.frames[0]= temp_x;
	new_y.frames[0] = temp_y;

	if(symetric)
		return (.5*(sum_x/(real)(x->n_frames) + sum_y/(real)y->n_frames));
	else
		return sum_x/(real)(x->n_frames);
}

MaxKernel::~MaxKernel()
{
}

//===================================================
//
//===================================================

HausdorffKernel::HausdorffKernel(Kernel* kernel_,real n_best_)
{
	kernel = kernel_;
	n_best = n_best_;
}

real HausdorffKernel::eval(Sequence *x, Sequence *y)
{
	Sequence new_x(1,x->frame_size);
	Sequence new_y(1,y->frame_size);
	real* temp_x = new_x.frames[0];
	real* temp_y = new_y.frames[0];

	
	//allocate memory for max value for y
	real* max_y = (real*)Allocator::sysAlloc(sizeof(real)*y->n_frames);
	real* max_x = (real*)Allocator::sysAlloc(sizeof(real)*x->n_frames);

	for(int j=0; j<y->n_frames;j++)
			max_y[j] = -INF;

	for(int i=0; i<x->n_frames;i++)
			max_x[i] = -INF;

	for(int i=0;i<x->n_frames;i++){

		new_x.frames[0] = x->frames[i];

		for(int j=0;j<y->n_frames;j++){
			
			new_y.frames[0] = y->frames[j];
			real val = kernel->eval(&new_x, &new_y);

			if (val> max_x[i])
				max_x[i] = val;
			
			if(val > max_y[j])
				max_y[j] = val;
		}
	}
	
	/*
	for(int j=0; j<y->n_frames;j++)
		max_y[j] *=-1;
	
	for(int i=0; i<x->n_frames;i++)
		max_x[i] *=-1;
		*/

	qsort(max_y,y->n_frames,sizeof(real),compar_real);
	
	qsort(max_x,x->n_frames,sizeof(real),compar_real);

	real val_x = .0;
	int n_x = (int)(n_best*x->n_frames);
	if(n_x == 0)
		n_x = 1;
	if(n_x > x->n_frames)
		n_x = x->n_frames;

	for(int i=0; i<n_x;i++)
		val_x += max_x[i];
	
	free(max_x);
	
	real val_y = .0;
	int n_y = (int)(n_best*y->n_frames);
	if(n_y == 0)
		n_y = 1;
	if(n_y > y->n_frames)
		n_y = y->n_frames;

	for(int j=0; j<n_y;j++)
		val_y += max_y[j];
	free(max_y);
	
	new_x.frames[0]= temp_x;
	new_y.frames[0] = temp_y;

	real ret = val_y/(real)n_y + val_x/(real)n_x;
	//message("ret: %g",ret);
	return (ret);
}

HausdorffKernel::~HausdorffKernel()
{
}
//===================================================
//
//===================================================

GMMKernel::GMMKernel(Kernel* kernel_, DiagonalGMM* cluster_)
{
	kernel = kernel_;
	cluster = cluster_;
}

real GMMKernel::eval(Sequence *x, Sequence *y)
{
	cluster->viterbiForward(x);
	Sequence x_best(cluster->best_gauss_per_frame->n_frames, cluster->best_gauss_per_frame->frame_size);
	x_best.copy(cluster->best_gauss_per_frame);

	cluster->viterbiForward(y);
	Sequence y_best(cluster->best_gauss_per_frame->n_frames, cluster->best_gauss_per_frame->frame_size);
	y_best.copy(cluster->best_gauss_per_frame);

	real sum = 0.;
	int n = 0;

	for (int i = 0; i<cluster->n_gaussians;i++){
			Sequence new_x(0, x->frame_size);
			Sequence new_y(0, y->frame_size);
			for (int j = 0; j< x->n_frames;j++)
				if ((int)x_best.frames[j][0] == i)
					new_x.addFrame(x->frames[j]);

			for (int j = 0; j< y->n_frames;j++)
				if ((int)y_best.frames[j][0] == i)
					new_y.addFrame(y->frames[j]);
			if (new_x.n_frames > 0 && new_y.n_frames>0){
				real val = kernel->eval(&new_x, &new_y);
				sum += val;
				n += 1;
			}
	}
	real	ret = .0;

	if (n > 0)
		ret = sum/(real)n;
	else
		warning("No comparable frames");


	return sum/(real)n;
}

GMMKernel::~GMMKernel()
{
}

//===================================================
//
//===================================================

ClusterKernel::ClusterKernel(Kernel* kernel_, int win_size_):MeanKernel(kernel_)
{
	win_size = win_size_;
}

real ClusterKernel::eval(Sequence *x, Sequence *y)
{
	int frame_size = x->frame_size;
	Sequence new_x(1,x->frame_size-1);
	Sequence new_y(1,y->frame_size-1);
	//to free correctely the memory
	real* temp_x = new_x.frames[0];
	real* temp_y = new_y.frames[0];
	//all kernel evaluations
	if(current_sequence != NULL)
		allocator->free(current_sequence);

	current_sequence = new(allocator)Sequence(x->n_frames,y->n_frames);
	
	//compute all possible values
	for(int i=0;i<x->n_frames;i++){
		//int x_target = (int)x->frames[i][frame_size-1];
		int x_target = (int)(x->frames[i][frame_size-1]+0.5);
		new_x.frames[0] = x->frames[i];
		for(int j=0;j<y->n_frames;j++){
			//int y_target = (int)y->frames[j][frame_size-1];
			int y_target = (int)(y->frames[j][frame_size-1]+0.5);
			new_y.frames[0] = y->frames[j];
			if(x_target == y_target){
				current_sequence->frames[i][j] =  kernel->eval(&new_x, &new_y);
				//message("%g ",current_sequence->frames[i][j]);
			}else
				current_sequence->frames[i][j] = -INF;
		}
	}

	//find the max for x frames
	real sum_x = 0;
	int n_x = 0;
	for(int i=0;i<x->n_frames;i++){
		real sum = 0;
		for(int w=0;w<win_size;w++)
			sum += current_sequence->frames[i][w];
		real best = sum;
		
		for(int j=1;j<y->n_frames-win_size+1;j++){
			sum = sum - current_sequence->frames[i][j-1] + current_sequence->frames[i][j+win_size-1];
			if(best < sum){
				best = sum;
			}
		}
		if(best > -INF+0.1){
		sum_x += best;
		n_x++;
		}
	}
	
	//find the max for y frames
	real sum_y = 0;
	int n_y = 0;
	for(int j=0;j<y->n_frames;j++){
		real sum = 0;
		for(int w=0;w<win_size;w++)
			sum += current_sequence->frames[w][j];
		real best = sum;
		
		for(int i=1;i<x->n_frames-win_size+1;i++){
			sum = sum - current_sequence->frames[i-1][j] + current_sequence->frames[i+win_size-1][j];
			if(best < sum){
				best = sum;
			}
		}
		if(best > -INF+0.1){
		sum_y += best;
		n_y++;
		}
	}

	new_x.frames[0]= temp_x;
	new_y.frames[0] = temp_y;
	if(sum_x == 0 && sum_y == 0){
		/*new_x.frames[0] = x->frames[x->n_frames/2];
		new_y.frames[0] = y->frames[y->n_frames/2];
		real rand_value = kernel->eval(&new_x, &new_y);
		new_x.frames[0]= temp_x;
		new_y.frames[0] = temp_y;*/

		warning("ClusterKernel:: No common frame");
		//message("Val: %g",rand_value);
		//return  rand_value;
		return 0.;
	}

	//message("ret_val: %g %g %g %g",sum_x, sum_y, n_x/(real)x->n_frames, n_y/(real)y->n_frames);
	real ret_val = (sum_x/(real)n_x +sum_y/(real)n_y);
	//real ret_val = (sum_x*n_x/(real)x->n_frames + sum_y*n_y/(real)y->n_frames);
	//real ret_val = (sum_x + sum_y);
	//message("Val: %g",ret_val);
	return ret_val;
}

ClusterKernel::~ClusterKernel()
{
}

//===================================================
//
//===================================================

DDKernel::DDKernel(Kernel* kernel_, int line_width_, int line_height_, int win_size_, bool symetric_)
{
	win_size = win_size_;
	kernel = kernel_;
	symetric = symetric_;
	x_cache = new(allocator)Sequence(line_width_, line_height_);
	if(! symetric)
		y_cache = new(allocator)Sequence(line_width_, line_height_);
}

void DDKernel::saveXFile(XFile* xf){
	IOAscii::saveSequence(xf, x_cache);
	if(!symetric)
		IOAscii::saveSequence(xf, y_cache);
}
real DDKernel::eval(Sequence *x, Sequence *y)
{
	compute_cache(x,y,x_cache);
	if(!symetric)
		compute_cache(y,x,y_cache);

	//average
	real sum = .0;
	for(int i=0;i<x_cache->n_frames;i++)
		for(int j=0;j<x_cache->frame_size;j++){
			sum += x_cache->frames[i][j];
			if(!symetric)
				sum += y_cache->frames[i][j];
		}
	real norm = (real)x_cache->n_frames*x_cache->frame_size;
	if(!symetric)
		norm *= 2.;
	return sum/norm;
}

void DDKernel::compute_cache(Sequence *x, Sequence *y, Sequence* cache){
	Sequence new_x(1,x->frame_size);
	Sequence new_y(1,y->frame_size);
	real* temp_x = new_x.frames[0];
	real* temp_y = new_y.frames[0];

	for(int i=0;i<cache->n_frames;i++){
		for(int j=0;j<cache->frame_size;j++){
			new_x.frames[0] = x->frames[i*cache->frame_size+j];
//printf("\ni: %d j: %d ",i,j);
			real max = -INF;
			for(int k=i-win_size;k<=i+win_size;k++){
				for(int l=j-win_size;l<=j+win_size;l++)
					if(k>-1 && l>-1 && k< cache->n_frames && l<cache->frame_size){
						new_y.frames[0] = y->frames[k*cache->frame_size+l];
						real tmp_val = kernel->eval(&new_x, &new_y);
						if(tmp_val > max)
							max = tmp_val;
//						printf(" (%d %d: %d) ",k,l,k*cache->frame_size+l);
					}
			}
			cache->frames[i][j]= max;
		}
	}

	new_x.frames[0]= temp_x;
	new_y.frames[0]= temp_y;
}

DDKernel::~DDKernel()
{
}

//===================================================
//
//===================================================

WinMaxKernel::WinMaxKernel(Kernel* kernel_, int  win_size_):MeanKernel(kernel_)
{
	win_size = win_size_;
	if (win_size < 1)
		error("WinMaxKernel: window size should be > 1");
}

real WinMaxKernel::eval(Sequence *x, Sequence *y)
{
	MeanKernel::eval(x,y);
	//find the max for x frames
	real sum_x = 0;
	for(int i=0;i<x->n_frames;i++){
		real sum = 0;
		for(int w=0;w<win_size;w++)
			sum += current_sequence->frames[i][w];
		real best = sum;
		
		for(int j=1;j<y->n_frames-win_size+1;j++){
			sum = sum - current_sequence->frames[i][j-1] + current_sequence->frames[i][j+win_size-1];
			if(best < sum){
				best = sum;
			}
		}
		sum_x += best;
	}
	
	//find the max for y frames
	real sum_y = 0;
	for(int j=0;j<y->n_frames;j++){
		real sum = 0;
		for(int w=0;w<win_size;w++)
			sum += current_sequence->frames[w][j];
		real best = sum;
		
		for(int i=1;i<x->n_frames-win_size+1;i++){
			sum = sum - current_sequence->frames[i-1][j] + current_sequence->frames[i+win_size-1][j];
			if(best < sum){
				best = sum;
			}
		}
		sum_y += best;
	}


	return (sum_x/(real)(x->n_frames) + sum_y/(real)y->n_frames);
}

WinMaxKernel::~WinMaxKernel()
{
}

GammaKernel::GammaKernel(Kernel* kernel_, int n_gamma_, int n_best_):MeanKernel(kernel_)
{
	n_best = n_best_;
	n_gamma = n_gamma_;
	x_sort = (Int_real**)allocator->alloc(n_gamma*sizeof(Int_real*));
	y_sort = (Int_real**)allocator->alloc(n_gamma*sizeof(Int_real*));
	for(int i=0;i<n_gamma;i++){
		x_sort[i] = (Int_real*)allocator->alloc(n_best*sizeof(Int_real));
		y_sort[i] = (Int_real*)allocator->alloc(n_best*sizeof(Int_real));
	}
}

void GammaKernel::sortByGamma(Sequence* x, Sequence* y)
{

	int frame_size = x->frame_size-n_gamma-1;
	for(int i=0;i<n_gamma;i++){
		//initialize
		for(int j=0;j<n_best;j++){
			x_sort[i][j].the_real = -INF;
			y_sort[i][j].the_real = -INF;
			x_sort[i][j].the_int = -1;
			y_sort[i][j].the_int = -1;
		}
		for(int j=0;j<x->n_frames;j++){
			if(x_sort[i][0].the_real < x->frames[j][i+frame_size]){
				x_sort[i][0].the_real = x->frames[j][i+frame_size];
				x_sort[i][0].the_int = j;
				qsort(x_sort[i],n_best,sizeof(Int_real),compar_int_real);
			}
		}
		for(int j=0;j<y->n_frames;j++){
			if(y_sort[i][0].the_real<y->frames[j][i+frame_size]){
				y_sort[i][0].the_real = y->frames[j][i+frame_size];
				y_sort[i][0].the_int = j;
				qsort(y_sort[i],n_best,sizeof(Int_real),compar_int_real);
			}
		}
	}

}

real GammaKernel::eval(Sequence *x, Sequence *y)
{
	int frame_size = x->frame_size;
	Sequence new_x(1,x->frame_size-1-n_gamma);
	Sequence new_y(1,y->frame_size-1-n_gamma);
	//to free correctely the memory
	real* temp_x = new_x.frames[0];
	real* temp_y = new_y.frames[0];
	//all kernel evaluations
	if(current_sequence != NULL)
		allocator->free(current_sequence);

	sortByGamma(x, y);
	current_sequence = new(allocator)Sequence(x->n_frames,y->n_frames);
	for(int i=0;i<x->n_frames;i++)
		for(int j=0;j<y->n_frames;j++)
			current_sequence->frames[i][j] = -INF;

	//compute all possible values for x
	real sum_x = .0;
	for(int i=0;i<x->n_frames;i++){
		int max_gamma = (int)(x->frames[i][frame_size-1]+0.5);
		new_x.frames[0] = x->frames[i];
		for(int j=0;j<n_best;j++){
			//int y_target = (int)y->frames[j][frame_size-1];
			int index = y_sort[max_gamma][j].the_int;
			new_y.frames[0] = y->frames[index];
			current_sequence->frames[i][index] =  kernel->eval(&new_x, &new_y);
			sum_x += current_sequence->frames[i][index];
			//message("%g ",current_sequence->frames[i][j]);
		}
	}

	real sum_y = .0;
	for(int i=0;i<y->n_frames;i++){
		int max_gamma = (int)(y->frames[i][frame_size-1]+0.5);
		new_y.frames[0] = y->frames[i];
		for(int j=0;j<n_best;j++){
			int index = x_sort[max_gamma][j].the_int;
			new_x.frames[0] = x->frames[index];
			if(current_sequence->frames[index][i]<=-INF)
				current_sequence->frames[index][i] =  kernel->eval(&new_x, &new_y);
			sum_y += current_sequence->frames[index][i];
		}
	}
	new_x.frames[0]= temp_x;
	new_y.frames[0] = temp_y;
	real ret_val = (sum_x/(real)(x->n_frames*n_best) +sum_y/(real)(y->n_frames*n_best));
	return ret_val;
}

GammaKernel::~GammaKernel()
{
}
}
