// Copyright (c) 2007 David Grangier
// Copyright (c) 2007 Samy Bengio
// 
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are 
// met: Redistributions of source code must retain the above copyright 
// notice, this list of conditions and the following disclaimer.
// Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the 
// documentation and/or other materials provided with the distribution.
// The name of the author may not be used to endorse or promote products
// derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, 
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.


#include "SMatrixCalculator.h"

namespace Torch {

SMatrixCalculator::SMatrixCalculator(SparseMatrix *mat_)
{
	mat=mat_;
	idf=(real*)allocator->alloc(sizeof(real)*mat->nc);
	std=(real*)allocator->alloc(sizeof(real)*mat->nc);
}

real SMatrixCalculator::meanLength()
{
	double l=0;
  for (int d=0;d<mat->nl;d++)
    for (int i=0;i<mat->lines[d].size;i++)
			l+=(double)mat->lines[d].frame[i].value;      
		
	l/=(double)mat->nl;
	return l;
}

void SMatrixCalculator::computeIdf()
{
	int *df=(int*)allocator->alloc(sizeof(int)*mat->nc);
	memset(df,0,sizeof(int)*mat->nc);

	for (int d=0;d<mat->nl;d++)
		for (int i=0;i<mat->lines[d].size;i++)
			df[mat->lines[d].frame[i].index]++;

	for (int t=0;t<mat->nc;t++)
		idf[t]=-log(((real)(df[t]))/((real)(mat->nl)));
}

void SMatrixCalculator::loadIdf(XFile *xfile)
{
	xfile->read(idf,sizeof(real)*mat->nc,1);
}

void SMatrixCalculator::saveIdf(XFile *xfile)
{
  xfile->write(idf,sizeof(real)*mat->nc,1);
}

void SMatrixCalculator::computeStd()
{
	int *df=(int*)allocator->alloc(sizeof(int)*mat->nc);
  memset(df,0,sizeof(int)*mat->nc);
	
	double *m=(double*)allocator->alloc(sizeof(double)*mat->nc);
  memset(m,0,sizeof(double)*mat->nc);

	// compute means
	for (int d=0;d<mat->nl;d++)
    for (int i=0;i<mat->lines[d].size;i++)
		{
			int t=mat->lines[d].frame[i].index;
      df[t]++;
			m[t]+=(double)mat->lines[d].frame[i].value;
		}
	for (int t=0;t<mat->nc;t++)
    m[t]/=(double)mat->nl;

  double *var=(double*)allocator->alloc(sizeof(double)*mat->nc);
  memset(var,0,sizeof(double)*mat->nc);

	// compute vars
  for (int d=0;d<mat->nl;d++)
    for (int i=0;i<mat->lines[d].size;i++)
		{
			int t=mat->lines[d].frame[i].index;
			double v=mat->lines[d].frame[i].value-m[t];
			var[t]+=v*v;
		}
	for (int t=0;t<mat->nc;t++)
	{
		var[t]+=m[t]*m[t]*(mat->nl-df[t]);
		var[t]/=(double)(mat->nl-1);
		std[t]=(real)sqrt(var[t]);
	}
}

void SMatrixCalculator::loadStd(XFile *xfile)
{
  xfile->read(std,sizeof(real)*mat->nc,1);
}

void SMatrixCalculator::saveStd(XFile *xfile)
{
  xfile->write(std,sizeof(real)*mat->nc,1);
}

void SMatrixCalculator::binaryWeight()
{
	for (int d=0;d<mat->nl;d++)
		for (int i = 0; i < mat->lines[d].size; i++)
		{
			mat->lines[d].frame[i].value = 1.0;
		}
}

void SMatrixCalculator::okapiLocalWeight(real k, real b)
{

	real meanLen = meanLength(); // mean doc length

  for (int d=0;d<mat->nl;d++)
	{
		// compute normalized doc length
		real l=0; // doc length
    for (int i = 0; i < mat->lines[d].size; i++)
			l += mat->lines[d].frame[i].value;
		real ndl = l / meanLen; // normalized length

		// compute weight for k, b, ndl
		for (int i = 0; i < mat->lines[d].size; i++) 
    {
			real tf = mat->lines[d].frame[i].value;
			real res = (k + 1) * tf / (k * ( (1-b) + b * ndl) + tf);
			mat->lines[d].frame[i].value = res;
    }
	}
}

void SMatrixCalculator::xIdf()
{
  for (int d=0;d<mat->nl;d++)
    for (int i=0;i<mat->lines[d].size;i++)
    {
      int t=mat->lines[d].frame[i].index;
      mat->lines[d].frame[i].value*=idf[t];
		}
}

void SMatrixCalculator::l2Norm(real *norm)
{
	for (int d=0;d<mat->nl;d++)
	{
		double dnorm=0;
    for (int i=0;i<mat->lines[d].size;i++)
    {
			real v=mat->lines[d].frame[i].value;
			dnorm+=v*v;
		}
		dnorm=sqrt(dnorm);
		norm[d]=(real)dnorm;
	}
}

void SMatrixCalculator::l1Norm(real *norm)
{
  for (int d=0;d<mat->nl;d++)
  {
    double dnorm=0;
    for (int i=0;i<mat->lines[d].size;i++)
    {
      real v=mat->lines[d].frame[i].value;
      dnorm+=fabs(v);
    }
    norm[d]=(real)dnorm;
  }
}

void SMatrixCalculator::scaleDoc(real *scaling)
{
  for (int d=0;d<mat->nl;d++)
    for (int i=0;i<mat->lines[d].size;i++)
      mat->lines[d].frame[i].value*=scaling[d];
}

void SMatrixCalculator::varNorm()
{
	for (int d=0;d<mat->nl;d++)
		for (int i=0;i<mat->lines[d].size;i++)
		{
      int t=mat->lines[d].frame[i].index;
      mat->lines[d].frame[i].value/=std[t];
		}
}

SMatrixCalculator::~SMatrixCalculator()
{}

}
