// Copyright (c) 2007 David Grangier
// Copyright (c) 2007 Samy Bengio
// 
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are 
// met: Redistributions of source code must retain the above copyright 
// notice, this list of conditions and the following disclaimer.
// Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the 
// documentation and/or other materials provided with the distribution.
// The name of the author may not be used to endorse or promote products
// derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, 
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.


#include "ImgConstraintPolicies.h"
#include "ImgConstraintDataset.h"
#include "ImgConstraint.h"
#include "Random.h"

namespace Torch {

ImgConstraintDataset::ImgConstraintDataset
 (SparseMatrix *queries_, SparseMatrix *relevance_, PADataset *images_,
 MarginPolicy *margin_, WeightPolicy *weight_)
{
	// data
	queries = queries_;
	relevance = relevance_;
	images = images_;
	// for shuffle
	n_que = queries_->nl;
	que_shuffle = (int*) allocator->alloc(sizeof(int) * n_que);
	// for margin/weighting
	margin = margin_;
	weight = weight_;
	// no current example
	current_example = NULL;
}

void ImgConstraintDataset::init()
{
	// shuffle queries
	cur_que_s = 0;
	Random::getShuffledIndices(que_shuffle, n_que);
	// empty stack
	while (example_stack->n_stacked_objects > 0)
		popExample();	
	// free current_example
	allocator->free(current_example);
	current_example=NULL;
}

PAExample *ImgConstraintDataset::nextExample()
{
  // query
  cur_que = que_shuffle[cur_que_s];
  svector *que = queries->lines + cur_que;
  svector *rel = relevance->lines + cur_que;
	// sample images
	sampleImg(rel);
	// get images
	images->popExample();
	PAExample *pos_img = images->getExample(cur_pos);
	images->pushExample();
	PAExample *neg_img = images->getExample(cur_neg);
	// margin / weight
	real epsilon = (margin) ? margin->getEpsilon(this) : 1.0;
	real w = (weight) ? weight->getWeight(this) : 1.0;
	// update current_example
	if (!current_example) 
		current_example = new (allocator) ImgConstraint();
	((ImgConstraint*)current_example)->set(que, pos_img, neg_img, epsilon, w);
	// update index
	cur_que_s = (cur_que_s + 1) % n_que;

	return current_example;
}

void ImgConstraintDataset::sampleImg(svector *rel)
{
  // sizes
  int n_img = relevance->nc;
  int n_rel = rel->size;
  // pos index
  cur_pos = rel->frame[Random::random() % n_rel].index;
	// neg index
  int neg = -1; // neg index, need to find the index of n th negative doc.
  int pos_index = 0;
  int neg_index = Random::random() % (n_img - n_rel);
  while (neg_index >= 0)
  {
    neg++;
    if ((pos_index >= n_rel)||(neg != rel->frame[pos_index].index))
      neg_index--;
    else
      pos_index++;
  }
	cur_neg = neg;	
}

ImgConstraintDataset::~ImgConstraintDataset()
{}

}
