// Copyright (c) 2007 David Grangier
// Copyright (c) 2007 Samy Bengio
// 
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are 
// met: Redistributions of source code must retain the above copyright 
// notice, this list of conditions and the following disclaimer.
// Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the 
// documentation and/or other materials provided with the distribution.
// The name of the author may not be used to endorse or promote products
// derived from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR 
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, 
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.


#include "ExpectedLikelihoodGaussian.h"
#include "log_add.h"

namespace Torch {

ExpectedLikelihoodGaussian::ExpectedLikelihoodGaussian(int dim_, real var_, bool feat_norm_)
{
	var = var_;
	feat_norm = feat_norm_;
	dim = dim_;

  mean_g = (real*) allocator->alloc(sizeof(real) * dim);
  var_g = (real*) allocator->alloc(sizeof(real) * dim);
}

// kernel between 2 GMMs
real ExpectedLikelihoodGaussian::eval(Sequence *x, Sequence *y)
{
	real res = log_eval(x, y);
	if (feat_norm)
		res -= 0.5 * (log_eval(x, x) + log_eval(y, y));
	res /= var;
	return exp(res);
}

real ExpectedLikelihoodGaussian::log_eval(Sequence *x, Sequence *y)
{
  // precompute f for all gaussians
  real *f_x = (real*) malloc(sizeof(real) * x->n_frames);
  real *f_y = (real*) malloc(sizeof(real) * y->n_frames);
	pre_compute_f(x, f_x);
	pre_compute_f(y, f_y);

	// log eval itself
	real log_sum = LOG_ZERO;

	real **frames_x = x->frames;
	real **frames_y = y->frames;
	real **last_x = frames_x + (x->n_frames);
  real **last_y = frames_y + (y->n_frames);

	while (frames_x != last_x)
	{
		while (frames_y != last_y)
		{
			real k_gauss = kGauss(*frames_x, *frames_y++, *f_x, *f_y++);
			log_sum = logAdd(log_sum, k_gauss);
		}
		frames_x++;
		f_x++;
		frames_y -= y->n_frames;
		f_y -= y->n_frames;
	}

	free(f_y);
	free(f_x - x->n_frames);

	return log_sum;
}

void ExpectedLikelihoodGaussian::pre_compute_f(Sequence *x, real *f_x)
{
	real **frames_x = x->frames;
	real **last_x = frames_x + (x->n_frames);

	while (frames_x != last_x)
	{
		*f_x++ = f(*frames_x + 1, *frames_x + (dim + 1));
		frames_x++;
	}
}

// log kernel between 2 Gaussians
real ExpectedLikelihoodGaussian::kGauss(real *x, real *y, real f_x, real f_y)
{
	// First Gaussian 
	real log_weight_x = *x;
	real *mean_x = x + 1;
	real *var_x	= x + (dim + 1);
	// Second Gaussian
  real log_weight_y = *y;
  real *mean_y = y + 1;
  real *var_y = y + (dim + 1);
	// Third Gaussian
	real *mean_z = mean_g;
	real *var_z = var_g;
	real *last_z = mean_z + dim;
	while (mean_z != last_z)
	{
		real inv_var_x = 1.0 / (*var_x++);
		real inv_var_y = 1.0 / (*var_y++);
		*mean_z++ = *mean_x++ * inv_var_x  +  *mean_y++ * inv_var_y;
		*var_z++ = 1.0 / ( inv_var_x + inv_var_y ) ;
	}

	real res = - ((real)dim)/2 * LOG_2_PI;
	res += log_weight_x + log_weight_y;
	res -= f_x;
  res -= f_y;
  res += g(mean_g, var_g);
	
	return res;
}

// f(x) = log( C^(1/2) exp( 1/2 u^T C^-1 u))
real ExpectedLikelihoodGaussian::f(real *mean, real *var)
{
	real res = 0;
	real *last_mean = mean + dim;
	while (mean != last_mean)
	{
		res += log(*var);
		res += *mean * *mean / *var++;
		mean++;
	}

	return (res/2);
}

// g(x) = log( C^(1/2) exp( 1/2 u^T C u))
real ExpectedLikelihoodGaussian::g(real *mean, real *var)
{
  real res = 0;
  real *last_mean = mean + dim;
  while (mean != last_mean)
  {
    res += log(*var);
    res += *mean * *mean * *var++;
		mean++;
  }

  return (res/2);
}


ExpectedLikelihoodGaussian::~ExpectedLikelihoodGaussian()
{
}

}
