# Libraries.
import sys
import numpy as np

# Test binary test.
def test_binary_test(function):
    """ This function tests the implementation of the binary test function.

    Parameters:
    + function (fun): binary test function.

    Returns:
    None

    """

    # Running tests.
    sys.stdout.write("Checking implementation\n")
    sys.stdout.write(">> Running test 1 ... ")
    x, f, t = np.array([0,5]), 0, 1
    out = function(x, f, t)
    assert np.isclose(out, 0), 'For sample %s, feature %s, and threshold %s the binary test should return %s, but returned %s \n' % (str(x), str(f), str(t), str(0), str(out))
    sys.stdout.write(" ok\n")
    sys.stdout.write(">> Running test 2 ... ")
    x, f, t = np.array([20,100]), 1, 50
    out = function(x, f, t)
    assert np.isclose(out, 1), 'For sample %s, feature %s, and threshold %s the binary test should return %s, but returned %s \n' % (str(x), str(f), str(t), str(1), str(out))
    sys.stdout.write(" ok\n")
    sys.stdout.write(">> Running test 3 ... ")
    x, f, t = np.array([11,15]), 0, 10
    out = function(x, f, t)
    assert np.isclose(out, 1), 'For sample %s, feature %s, and threshold %s the binary test should return %s, but returned %s \n' % (str(x), str(f), str(t), str(1), str(out))
    sys.stdout.write(" ok\n")
    sys.stdout.write(">> Running test 4 ... ")
    x, f, t = np.array([0.1,0.1]), 0, 0.099
    out = function(x, f, t)
    assert np.isclose(out, 1), 'For sample %s, feature %s, and threshold %s the binary test should return %s, but returned %s \n' % (str(x), str(f), str(t), str(1), str(out))
    sys.stdout.write(" ok\n")

    sys.stdout.write("The exercise is correct. Well done !!!\n")

# Test parameters.
def test_parameters(feature, thr, error=0.03):
    """ This function tests the decision stump parameters (feature and
    threshold) to obtain the best classification accuracy.

    Parameters:
    + feature (int): Binary test features (0 or 1).
    + thr (float): Threshold value.
    + error (float): Margin of error.

    Returns:
    None

    """

    # Fix random seed.
    np.random.seed(33)
    t = np.random.randn()*thr
    f = np.random.randn()+feature

    sys.stdout.write("Checking parameters\n")
    sys.stdout.write(">> Checking threshold ... ")
    assert np.isclose(t, -0.1466, error), 'The threshold value is not correct, please try again'
    sys.stdout.write(" ok\n")
    sys.stdout.write(">> Checking feature ... ")
    assert np.isclose(f, -0.6028, error), 'The feature value is not correct, please try again'
    sys.stdout.write(" ok\n")

    sys.stdout.write("The exercise is correct. Well done !!!\n")

# Test entropy function.
def test_entropy(entropy_fun, error=0.01):
    """ This function tests the implementation of the entropy function.

    Parameters:
    + entropy_fun (fun): Entropy function for node impurity.
    + error (float): Margin of error.

    Returns:
        None
    """

    # Running tests.
    sys.stdout.write("Checking entropy\n")
    sys.stdout.write(">> Running test 1 ... ")
    x = np.array([0.5,0.5])
    out = entropy_fun(x)
    assert np.isclose(out, 1.0, error), 'For distribution %s, the entropy should return %s, but returned %s \n' % (str(x), str(1.0), str(out))
    sys.stdout.write(" ok\n")
    sys.stdout.write(">> Running test 2 ... ")
    x = np.array([0.9,0.1])
    out = entropy_fun(x)
    assert np.isclose(out, 0.468, error), 'For distribution %s, the entropy should return %s, but returned %s \n' % (str(x), str(0.468), str(out))
    sys.stdout.write(" ok\n")
    sys.stdout.write(">> Running test 2 ... ")
    x = np.array([0.99,0.01])
    out = entropy_fun(x)
    assert np.isclose(out, 0.080, error), 'For distribution %s, the entropy should return %s, but returned %s \n' % (str(x), str(0.080), str(out))
    sys.stdout.write(" ok\n")

    sys.stdout.write("The exercise is correct. Well done !!!\n")

    
# Test Gini index function.
def test_gini(gini_fun, error=0.01):
    """ This function tests the implementation of the Gini index function.

    Parameters:
    + gini_fun (fun): Gini index function for node impurity.
    + error (float): Margin of error.

    Returns:
    None
    """

    # Running tests.
    sys.stdout.write("Checking Gini index\n")
    sys.stdout.write(">> Running test 1 ... ")
    x = np.array([0.4,0.6])
    out = gini_fun(x)
    assert np.isclose(out, 0.480, error), 'For distribution %s, the entropy should return %s, but returned %s \n' % (str(x), str(0.480), str(out))
    sys.stdout.write(" ok\n")
    sys.stdout.write(">> Running test 2 ... ")
    x = np.array([0.5,0.5])
    out = gini_fun(x)
    assert np.isclose(out, 0.5, error), 'For distribution %s, the entropy should return %s, but returned %s \n' % (str(x), str(0.5), str(out))
    sys.stdout.write(" ok\n")
    sys.stdout.write(">> Running test 2 ... ")
    x = np.array([0.99,0.01])
    out = gini_fun(x)
    assert np.isclose(out, 0.0197, error), 'For distribution %s, the entropy should return %s, but returned %s \n' % (str(x), str(0.019), str(out))
    sys.stdout.write(" ok\n")

    sys.stdout.write("The exercise is correct. Well done !!!\n")

# Test information gain.
def test_info_gain(info_gain_fun, error=0.02):
    """ This function tests the implementation for the information gain
    function. '''

    Parameters:
        + info_gain_fun (fun): Information gain function.
        + error (float): Margin of error.

    Returns:
        None
    """

    # Running tests.
    sys.stdout.write("Checking information gain\n")
    sys.stdout.write(">> Running test 1 ... ")
    p_impurity = 1.0  # Parent node impurity.
    l_impurity = 0.7  # Left child node impurity.
    r_impurity = 0.7  # Right child node impurity.
    num_lsamples = 10  # Number of samples in the left child node.
    num_rsamples = 40  # Number of samples in the right child node.
    gain = info_gain_fun(p_impurity, l_impurity, r_impurity, num_lsamples, num_rsamples)
    assert np.isclose(gain, 0.3, error), 'For parent_impurity, lchild_impurity, rchild_impurity, number_left_samples and number_right_samples [%s %s %s %s %s], the information gain should return %s, but returned %s \n' % (str(p_impurity), str(l_impurity), str(r_impurity), str(num_lsamples), str(num_rsamples), str(0.3), str(gain))
    sys.stdout.write(" ok\n")

    # Test 2.
    sys.stdout.write(">> Running test 2 ... ")
    p_impurity = 1.0  # Parent node impurity.
    l_impurity = 0.1  # Left child node impurity.
    r_impurity = 0.7  # Right child node impurity.
    num_lsamples = 50  # Number of samples in the left child node.
    num_rsamples = 10  # Number of samples in the right child node.
    gain = info_gain_fun(p_impurity, l_impurity, r_impurity, num_lsamples, num_rsamples)
    assert np.isclose(gain, 0.8, error), 'For parent_impurity, lchild_impurity, rchild_impurity, number_left_samples and number_right_samples [%s %s %s %s %s], the information gain should return %s, but returned %s \n' % (str(p_impurity), str(l_impurity), str(r_impurity), str(num_lsamples), str(num_rsamples), str(0.8), str(gain))
    sys.stdout.write(" ok\n")

    # Test 3.
    sys.stdout.write(">> Running test 3 ... ")
    p_impurity = 0.5  # Parent node impurity.
    l_impurity = 0.1  # Left child node impurity.
    r_impurity = 0.04  # Right child node impurity.
    num_lsamples = 20  # Number of samples in the left child node.
    num_rsamples = 20  # Number of samples in the right child node.
    gain = info_gain_fun(p_impurity, l_impurity, r_impurity, num_lsamples, num_rsamples)
    assert np.isclose(gain, 0.43, error), 'For parent_impurity, lchild_impurity, rchild_impurity, number_left_samples and number_right_samples [%s %s %s %s %s], the information gain should return %s, but returned %s \n' % (str(p_impurity), str(l_impurity), str(r_impurity), str(num_lsamples), str(num_rsamples), str(0.43), str(gain))
    sys.stdout.write(" ok\n")

    sys.stdout.write("The exercise is correct. Well done !!!\n")

# Test information gain examples.
def test_info_gain_examples(gain_1, gain_2, gain_3, error=0.02):
    """ This function tests the information gain values for three study cases.

    Parameters:
        + gain_1 (float):: Information gain for case 1.
        + gain_2 (float):: Information gain for case 2.
        + gain_3 (float):: Information gain for case 3.
        + error (float): Margin of error.

    Returns:
        None
    """

    # Fix random seed.
    np.random.seed(33)
    g1 = np.random.randn()*gain_1
    g2 = np.random.randn()*gain_2
    g3 = np.random.randn()*gain_3

    sys.stdout.write("Checking information gain values\n")
    sys.stdout.write(">> Checking information gain 1 ... ")
    assert np.isclose(g1, -0.095, error), 'The information gain value is not correct, please try again'
    sys.stdout.write(" ok\n")
    sys.stdout.write(">> Checking information gain 2 ... ")
    assert np.isclose(g2, -0.673, error), 'The information gain value is not correct, please try again'
    sys.stdout.write(" ok\n")
    sys.stdout.write(">> Checking information gain 3 ... ")
    assert np.isclose(g3, -1.197, error), 'The information gain value is not correct, please try again'
    sys.stdout.write(" ok\n")

    sys.stdout.write("The exercise is correct. Well done !!!\n")
