# Libraries.
import time
import operator
import numpy as np
import pickle as pk
import matplotlib.pyplot as plt
from IPython.display import display
from sklearn.metrics import mean_squared_error, r2_score
from collections import OrderedDict

# Sort dictionary.
def sort_dictionary(data):
    ''' This function sorts the input dictionary. '''
    return OrderedDict(sorted(data.items(), key=lambda t: t[0]))

# Sort data values.
def sort_data(x, y):
    """This function sorts the data (samples) according 
    to x values. """
    sort_axis = operator.itemgetter(0)
    sorted_zip = sorted(zip(x,y), key=sort_axis)
    x, y = zip(*sorted_zip)
    return x, y

# Show dataset samples.
def show_dataset(data, title='Train Data', color='r'):
    """This function shows the dataset samples (x,y). """
    x, y = sort_data(data['x'], data['y'])
    plt.figure(figsize=(10, 5))
    plt.plot(x, y, 'o', color=color)
    plt.grid()
    plt.title(title, fontsize=18)
    plt.xlabel('x', fontsize=14)
    plt.ylabel('y', fontsize=14)
    plt.show()
    
# Colors.
def colors(num_max=12, seed=22):
    """This functions creates a palette of colors. """
    np.random.seed(seed)
    colors = np.random.rand(num_max, 3)
    # Reformat colors.
    colors[0,:] = [0.90, 0.01, 0.01]
    colors[1,:] = [0.01, 0.71, 0.01]
    colors[2,:] = [0.01, 0.01, 0.91]
    colors[3,:] = [0.01, 0.01, 0.01]
    colors[4,:] = [0.81, 0.01, 0.81]
    colors[5,:] = [0.81, 0.81, 0.01]
    colors[6,:] = [0.01, 0.81, 0.81]
    colors[7,:] = [0.51, 0.51, 0.11]
    return colors

# Show predictions..        
def show_predictions(preds):
    """This function shows the predictions plots for train 
    and test samples."""

    # Pre-defined colors.
    cls = colors()

    # Sort predictions -dictionary-.
    preds = sort_dictionary(preds)

    # Train predictions.
    plt.figure(figsize=(15,15))
    plt.subplot(211)
    for k, d in enumerate(preds):
        x, y = sort_data(preds[d]['train']['x'], preds[d]['train']['y'])
        plt.plot(x, y, color=cls[k,:], label='Degree='+str(d), linewidth=4)
    plt.grid()
    plt.legend(fontsize=15)
    plt.title('Train Data', fontsize=18)
    plt.xlabel('x', fontsize=14)
    plt.ylabel('y', fontsize=14)
    axes = plt.gca()
    axes.set_ylim([-0.4,1.4])
       
    # Test predictions.
    plt.subplot(212)
    for k, d in enumerate(preds):
        x, y = sort_data(preds[d]['test']['x'], preds[d]['test']['y'])
        plt.plot(x, y, color=cls[k,:], label='Degree='+str(d), linewidth=4)
    plt.grid()
    plt.legend(fontsize=15)
    plt.title('Test Data', fontsize=18)
    plt.xlabel('x', fontsize=14)
    plt.ylabel('y', fontsize=14)
    axes = plt.gca()
    axes.set_ylim([-0.4,1.4])
    plt.show()   
    
# Show predictions -array-.
def show_predictions_array(preds, data='test'):
    """This function shows the predictions plots in terms 
    of the number of samples and polynomial degree."""

    # Colors.
    cls = colors()

    # Sort predictions -dictionary-.
    preds = sort_dictionary(preds)

    # Number of plots.
    max_plots = len(preds)

    # Show plots.
    v=0
    plt.figure(figsize=(30,80))
    for n in preds:
        # Update subplot index.
        v = v+1
        ax1 = plt.subplot(max_plots,2,v)
        # Sort current predictions -dictionary-.
        c_preds = sort_dictionary(preds[n])
        # Plot predictions.
        for k, d in enumerate(c_preds):
            x, y = sort_data(c_preds[d][data]['x'], c_preds[d][data]['y'])
            plt.plot(x, y, color=cls[k,:], label='Degree='+str(d), linewidth=4)
        plt.grid()
        plt.legend(fontsize=24)
        plt.title('Num. train samples: '+str(n), fontsize=28)
        plt.xlabel('x', fontsize=24)
        plt.ylabel('y', fontsize=24)
        axes = plt.gca()
        axes.set_ylim([-0.4,1.4])
        axes.xaxis.set_tick_params(labelsize=18)
        axes.yaxis.set_tick_params(labelsize=18)
    plt.show()    

# Show predictions -array-.
def show_rmse_array(preds):
    """This function shows the predictions plots in terms 
    of the number of samples and polynomial degree."""

    # Colors.
    cls = colors()

    # Sort predictions -dictionary-.
    preds = sort_dictionary(preds)

    # Number of plots.
    max_plots = len(preds)

    # Show plots.
    v=0
    plt.figure(figsize=(20,50))
    for n in preds:
        # Update subplot index.
        v = v+1
        ax1 = plt.subplot(max_plots,2,v)
        # Sort current predictions -dictionary-.
        c_preds = sort_dictionary(preds[n])
        # Plot predictions.
        train_vec = []
        test_vec = []
        x_vec = []
        for k, d in enumerate(c_preds):
            x_vec.append(d)
            train_vec.append(c_preds[d]['train']['rmse'])
            test_vec.append(c_preds[d]['test']['rmse'])
        plt.plot(x_vec, train_vec, '*-', color=cls[0,:], label='Train', linewidth=4, markersize=20)
        plt.plot(x_vec, test_vec, '*-', color=cls[3,:], label='Test', linewidth=4, markersize=20)
        plt.grid()
        plt.legend(fontsize=24)
        plt.title('Num. train samples: '+str(n), fontsize=28)
        plt.xlabel('Polynomial Degree', fontsize=24)
        plt.ylabel('RMSE', fontsize=24)
        axes = plt.gca()
        axes.set_ylim([0.0,3.0])
        axes.xaxis.set_tick_params(labelsize=18)
        axes.yaxis.set_tick_params(labelsize=18)
    plt.show()    
